/* packet-ipv6.c
 * Routines for IPv6 packet disassembly
 *
 * $Id: packet-ipv6.c 18830 2006-08-03 20:24:24Z jake $
 *
 * Wireshark - Network traffic analyzer
 * By Gerald Combs <gerald@wireshark.org>
 * Copyright 1998 Gerald Combs
 *
 * SHIM6 support added by Matthijs Mekking <matthijs@NLnetLabs.nl>
 *
 * MobileIPv6 support added by Tomislav Borosa <tomislav.borosa@siemens.hr>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <string.h>
#include <math.h>
#include <stdio.h>
#include <glib.h>
#include <epan/packet.h>
#include "packet-ipsec.h"
#include "packet-ipv6.h"
#include <epan/ip_opts.h>
#include <epan/addr_resolv.h>
#include <epan/prefs.h>
#include <epan/reassemble.h>
#include <epan/ipproto.h>
#include <epan/ipv6-utils.h>
#include <epan/etypes.h>
#include <epan/ppptypes.h>
#include <epan/aftypes.h>
#include <epan/nlpid.h>
#include <epan/arcnet_pids.h>
#include <epan/in_cksum.h>

/*
 * NOTE: ipv6.nxt is not very useful as we will have chained header.
 * now testing ipv6.final, but it raises SEGV.
#define TEST_FINALHDR
 */

static int proto_ipv6 = -1;
static int hf_ipv6_version = -1;
static int hf_ipv6_class = -1;
static int hf_ipv6_flow = -1;
static int hf_ipv6_plen = -1;
static int hf_ipv6_nxt = -1;
static int hf_ipv6_hlim = -1;
static int hf_ipv6_src = -1;
static int hf_ipv6_dst = -1;
static int hf_ipv6_addr = -1;
#ifdef TEST_FINALHDR
static int hf_ipv6_final = -1;
#endif
static int hf_ipv6_fragments = -1;
static int hf_ipv6_fragment = -1;
static int hf_ipv6_fragment_overlap = -1;
static int hf_ipv6_fragment_overlap_conflict = -1;
static int hf_ipv6_fragment_multiple_tails = -1;
static int hf_ipv6_fragment_too_long_fragment = -1;
static int hf_ipv6_fragment_error = -1;
static int hf_ipv6_reassembled_in = -1;

static int hf_ipv6_mipv6_type = -1;
static int hf_ipv6_mipv6_length = -1;
static int hf_ipv6_mipv6_home_address = -1;

static int hf_ipv6_shim6 = -1;
static int hf_ipv6_shim6_nxt = -1;
static int hf_ipv6_shim6_len = -1;
static int hf_ipv6_shim6_p = -1;
/* context tag is 49 bits, cannot be used for filter yet */
static int hf_ipv6_shim6_ct = -1; 
static int hf_ipv6_shim6_type = -1; 
static int hf_ipv6_shim6_proto = -1; 
static int hf_ipv6_shim6_csum = -1; 
static int hf_ipv6_shim6_inonce = -1; /* also for request nonce */
static int hf_ipv6_shim6_rnonce = -1; 
static int hf_ipv6_shim6_precvd = -1; 
static int hf_ipv6_shim6_psent = -1; 
static int hf_ipv6_shim6_reap = -1; 
static int hf_ipv6_shim6_opt_type = -1; 
static int hf_ipv6_shim6_opt_len = -1; 
static int hf_ipv6_shim6_opt_critical = -1; 
static int hf_ipv6_shim6_opt_loclist = -1; 
static int hf_ipv6_shim6_opt_locnum = -1; 
static int hf_ipv6_shim6_opt_elemlen = -1; 
static int hf_ipv6_shim6_opt_fii = -1; 

static gint ett_ipv6 = -1;
static gint ett_ipv6_fragments = -1;
static gint ett_ipv6_fragment  = -1;

static const fragment_items ipv6_frag_items = {
	&ett_ipv6_fragment,
	&ett_ipv6_fragments,
	&hf_ipv6_fragments,
	&hf_ipv6_fragment,
	&hf_ipv6_fragment_overlap,
	&hf_ipv6_fragment_overlap_conflict,
	&hf_ipv6_fragment_multiple_tails,
	&hf_ipv6_fragment_too_long_fragment,
	&hf_ipv6_fragment_error,
	&hf_ipv6_reassembled_in,
	"fragments"
};

static dissector_handle_t data_handle;

static dissector_table_t ip_dissector_table;

/* Reassemble fragmented datagrams */
static gboolean ipv6_reassemble = TRUE;

#ifndef offsetof
#define	offsetof(type, member)	((size_t)(&((type *)0)->member))
#endif

/*
 * defragmentation of IPv6
 */
static GHashTable *ipv6_fragment_table = NULL;
static GHashTable *ipv6_reassembled_table = NULL;

void
capture_ipv6(const guchar *pd, int offset, int len, packet_counts *ld)
{
  guint8 nxt;
  int advance;

  if (!BYTES_ARE_IN_FRAME(offset, len, 4+4+16+16)) {
    ld->other++;
    return;
  }
  nxt = pd[offset+6];		/* get the "next header" value */
  offset += 4+4+16+16;		/* skip past the IPv6 header */

again:
   switch (nxt) {
   case IP_PROTO_HOPOPTS:
   case IP_PROTO_ROUTING:
   case IP_PROTO_DSTOPTS:
     if (!BYTES_ARE_IN_FRAME(offset, len, 2)) {
       ld->other++;
       return;
     }
     nxt = pd[offset];
     advance = (pd[offset+1] + 1) << 3;
     if (!BYTES_ARE_IN_FRAME(offset, len, advance)) {
       ld->other++;
       return;
     }
     offset += advance;
     goto again;
   case IP_PROTO_FRAGMENT:
     if (!BYTES_ARE_IN_FRAME(offset, len, 2)) {
       ld->other++;
       return;
     }
     nxt = pd[offset];
     advance = 8;
     if (!BYTES_ARE_IN_FRAME(offset, len, advance)) {
       ld->other++;
       return;
     }
     offset += advance;
     goto again;
   case IP_PROTO_AH:
     if (!BYTES_ARE_IN_FRAME(offset, len, 2)) {
       ld->other++;
       return;
     }
     nxt = pd[offset];
     advance = 8 + ((pd[offset+1] - 1) << 2);
     if (!BYTES_ARE_IN_FRAME(offset, len, advance)) {
       ld->other++;
       return;
     }
     offset += advance;
     goto again;
   case IP_PROTO_SHIM6:
     if (!BYTES_ARE_IN_FRAME(offset, len, 2)) {
       ld->other++;
       return;
     }
     nxt = pd[offset];
     advance = (pd[offset+1] + 1) << 3;
     if (!BYTES_ARE_IN_FRAME(offset, len, advance)) {
       ld->other++;
       return;
     }
     offset += advance;
     goto again;
   }

  switch(nxt) {
    case IP_PROTO_SCTP:
      ld->sctp++;
      break;
    case IP_PROTO_TCP:
      ld->tcp++;
      break;
    case IP_PROTO_UDP:
    case IP_PROTO_UDPLITE:
      ld->udp++;
      break;
    case IP_PROTO_ICMP:
    case IP_PROTO_ICMPV6:	/* XXX - separate counters? */
      ld->icmp++;
      break;
    case IP_PROTO_OSPF:
      ld->ospf++;
      break;
    case IP_PROTO_GRE:
      ld->gre++;
      break;
    case IP_PROTO_VINES:
      ld->vines++;
      break;
    default:
      ld->other++;
  }
}

static void
ipv6_reassemble_init(void)
{
  fragment_table_init(&ipv6_fragment_table);
  reassembled_table_init(&ipv6_reassembled_table);
}

static int
dissect_routing6(tvbuff_t *tvb, int offset, proto_tree *tree) {
    struct ip6_rthdr rt;
    guint len;
    proto_tree *rthdr_tree;
    proto_item *ti;
    guint8 buf[sizeof(struct ip6_rthdr0) + sizeof(struct e_in6_addr) * 23];

    tvb_memcpy(tvb, (guint8 *)&rt, offset, sizeof(rt));
    len = (rt.ip6r_len + 1) << 3;

    if (tree) {
	/* !!! specify length */
	ti = proto_tree_add_text(tree, tvb, offset, len,
	    "Routing Header, Type %u", rt.ip6r_type);
	rthdr_tree = proto_item_add_subtree(ti, ett_ipv6);

	proto_tree_add_text(rthdr_tree, tvb,
	    offset + offsetof(struct ip6_rthdr, ip6r_nxt), 1,
	    "Next header: %s (0x%02x)", ipprotostr(rt.ip6r_nxt), rt.ip6r_nxt);
	proto_tree_add_text(rthdr_tree, tvb,
	    offset + offsetof(struct ip6_rthdr, ip6r_len), 1,
	    "Length: %u (%d bytes)", rt.ip6r_len, len);
	proto_tree_add_text(rthdr_tree, tvb,
	    offset + offsetof(struct ip6_rthdr, ip6r_type), 1,
	    "Type: %u", rt.ip6r_type);
	proto_tree_add_text(rthdr_tree, tvb,
	    offset + offsetof(struct ip6_rthdr, ip6r_segleft), 1,
	    "Segments left: %u", rt.ip6r_segleft);

	if (rt.ip6r_type == 0 && len <= sizeof(buf)) {
	    struct e_in6_addr *a;
	    int n;
	    struct ip6_rthdr0 *rt0;

	    tvb_memcpy(tvb, buf, offset, len);
	    rt0 = (struct ip6_rthdr0 *)buf;
	    for (a = rt0->ip6r0_addr, n = 0;
		 a < (struct e_in6_addr *)(buf + len);
		 a++, n++) {
		proto_tree_add_text(rthdr_tree, tvb,
		    offset + offsetof(struct ip6_rthdr0, ip6r0_addr) + n * sizeof(struct e_in6_addr),
		    sizeof(struct e_in6_addr),
#ifdef INET6
		    "address %d: %s (%s)",
		    n, get_hostname6(a), ip6_to_str(a)
#else
		    "address %d: %s", n, ip6_to_str(a)
#endif
		    );
	    }
	}
	if (rt.ip6r_type == 2) {
	    proto_tree_add_ipv6(rthdr_tree, hf_ipv6_mipv6_home_address,
				       tvb, offset + 8, 16,
				       tvb_get_ptr(tvb, offset + 8, 16));
	}
    }

    return len;
}

static int
dissect_frag6(tvbuff_t *tvb, int offset, packet_info *pinfo, proto_tree *tree,
    guint16 *offlg, guint32 *ident) {
    struct ip6_frag frag;
    int len;
    proto_item *ti;
    proto_tree *rthdr_tree;

    tvb_memcpy(tvb, (guint8 *)&frag, offset, sizeof(frag));
    len = sizeof(frag);
    frag.ip6f_offlg = g_ntohs(frag.ip6f_offlg);
    frag.ip6f_ident = g_ntohl(frag.ip6f_ident);
    *offlg = frag.ip6f_offlg;
    *ident = frag.ip6f_ident;
    if (check_col(pinfo->cinfo, COL_INFO)) {
	col_add_fstr(pinfo->cinfo, COL_INFO,
	    "IPv6 fragment (nxt=%s (0x%02x) off=%u id=0x%x)",
	    ipprotostr(frag.ip6f_nxt), frag.ip6f_nxt,
	    frag.ip6f_offlg & IP6F_OFF_MASK, frag.ip6f_ident);
    }
    if (tree) {
	   ti = proto_tree_add_text(tree, tvb, offset, len,
			   "Fragmentation Header");
	   rthdr_tree = proto_item_add_subtree(ti, ett_ipv6);

	   proto_tree_add_text(rthdr_tree, tvb,
			 offset + offsetof(struct ip6_frag, ip6f_nxt), 1,
			 "Next header: %s (0x%02x)",
			 ipprotostr(frag.ip6f_nxt), frag.ip6f_nxt);

#if 0
	   proto_tree_add_text(rthdr_tree, tvb,
			 offset + offsetof(struct ip6_frag, ip6f_reserved), 1,
			 "Reserved: %u",
			 frag.ip6f_reserved);
#endif

	   proto_tree_add_text(rthdr_tree, tvb,
			 offset + offsetof(struct ip6_frag, ip6f_offlg), 2,
			 "Offset: %u",
			 frag.ip6f_offlg & IP6F_OFF_MASK);

	   proto_tree_add_text(rthdr_tree, tvb,
			 offset + offsetof(struct ip6_frag, ip6f_offlg), 2,
			 "More fragments: %s",
				frag.ip6f_offlg & IP6F_MORE_FRAG ?
				"Yes" : "No");

	   proto_tree_add_text(rthdr_tree, tvb,
			 offset + offsetof(struct ip6_frag, ip6f_ident), 4,
			 "Identification: 0x%08x",
			 frag.ip6f_ident);
    }
    return len;
}

static int
dissect_mipv6_hoa(tvbuff_t *tvb, proto_tree *dstopt_tree, int offset)
{
    int len = 0;

    proto_tree_add_uint_format(dstopt_tree, hf_ipv6_mipv6_type, tvb,
	offset + len, 1,
	tvb_get_guint8(tvb, offset + len),
	"Option Type: %u (0x%02x) - Home Address Option",
	tvb_get_guint8(tvb, offset + len),
	tvb_get_guint8(tvb, offset + len));
    len += 1;

    proto_tree_add_uint(dstopt_tree, hf_ipv6_mipv6_length, tvb, offset + len,
	1, tvb_get_guint8(tvb, offset + len));
    len += 1;

    proto_tree_add_ipv6(dstopt_tree, hf_ipv6_mipv6_home_address, tvb,
	offset + len, 16, tvb_get_ptr(tvb, offset + len, 16));
    len += 16;
    return len;
}

static const value_string rtalertvals[] = {
    { IP6OPT_RTALERT_MLD, "MLD" },
    { IP6OPT_RTALERT_RSVP, "RSVP" },
    { 0, NULL },
};

/* Like "dissect_ip_tcp_options()", but assumes the length of an option
   *doesn't* include the type and length bytes. */
void
dissect_ipv6_options(tvbuff_t *tvb, int offset, guint length,
			const ip_tcp_opt *opttab, int nopts, int eol,
			packet_info *pinfo, proto_tree *opt_tree)
{
  guchar            opt;
  const ip_tcp_opt *optp;
  opt_len_type      len_type;
  unsigned int      optlen;
  const char       *name;
  char              name_str[7+1+1+2+2+1+1];	/* "Unknown (0x%02x)" */
  void            (*dissect)(const struct ip_tcp_opt *, tvbuff_t *,
				int, guint, packet_info *, proto_tree *);
  guint             len;

  while (length > 0) {
    opt = tvb_get_guint8(tvb, offset);
    for (optp = &opttab[0]; optp < &opttab[nopts]; optp++) {
      if (optp->optcode == opt)
        break;
    }
    if (optp == &opttab[nopts]) {
      /* We assume that the only NO_LENGTH options are Pad1 options,
         so that we can treat unknown options as VARIABLE_LENGTH with a
	 minimum of 0, and at least be able to move on to the next option
	 by using the length in the option. */
      optp = NULL;	/* indicate that we don't know this option */
      len_type = VARIABLE_LENGTH;
      optlen = 0;
      g_snprintf(name_str, sizeof name_str, "Unknown (0x%02x)", opt);
      name = name_str;
      dissect = NULL;
    } else {
      len_type = optp->len_type;
      optlen = optp->optlen;
      name = optp->name;
      dissect = optp->dissect;
    }
    --length;      /* account for type byte */
    if (len_type != NO_LENGTH) {
      /* Option has a length. Is it in the packet? */
      if (length == 0) {
        /* Bogus - packet must at least include option code byte and
           length byte! */
        proto_tree_add_text(opt_tree, tvb, offset,      1,
              "%s (length byte past end of options)", name);
        return;
      }
      len = tvb_get_guint8(tvb, offset + 1);  /* total including type, len */
      --length;    /* account for length byte */
      if (len > length) {
        /* Bogus - option goes past the end of the header. */
        proto_tree_add_text(opt_tree, tvb, offset,      length,
              "%s (option length = %u byte%s says option goes past end of options)",
	      name, len, plurality(len, "", "s"));
        return;
      } else if (len_type == FIXED_LENGTH && len != optlen) {
        /* Bogus - option length isn't what it's supposed to be for this
           option. */
        proto_tree_add_text(opt_tree, tvb, offset,      2 + len,
              "%s (with option length = %u byte%s; should be %u)", name,
              len, plurality(len, "", "s"), optlen);
        return;
      } else if (len_type == VARIABLE_LENGTH && len < optlen) {
        /* Bogus - option length is less than what it's supposed to be for
           this option. */
        proto_tree_add_text(opt_tree, tvb, offset,      2 + len,
              "%s (with option length = %u byte%s; should be >= %u)", name,
              len, plurality(len, "", "s"), optlen);
        return;
      } else {
        if (optp == NULL) {
          proto_tree_add_text(opt_tree, tvb, offset,    2 + len, "%s (%u byte%s)",
				name, len, plurality(len, "", "s"));
        } else {
          if (dissect != NULL) {
            /* Option has a dissector. */
            (*dissect)(optp, tvb, offset,          2 + len, pinfo, opt_tree);
          } else {
            /* Option has no data, hence no dissector. */
            proto_tree_add_text(opt_tree, tvb, offset,  2 + len, "%s", name);
          }
        }
        offset += 2 + len;
      }
      length -= len;
    } else {
      proto_tree_add_text(opt_tree, tvb, offset,      1, "%s", name);
      offset += 1;
    }
    if (opt == eol)
      break;
  }
}

static int
dissect_opts(tvbuff_t *tvb, int offset, proto_tree *tree, const char *optname)
{
    struct ip6_ext ext;
    int len;
    proto_tree *dstopt_tree;
    proto_item *ti;
    gint p;
    guint8 tmp;
    int mip_offset = 0, delta = 0;

    tvb_memcpy(tvb, (guint8 *)&ext, offset, sizeof(ext));
    len = (ext.ip6e_len + 1) << 3;

    if (tree) {
	/* !!! specify length */
	ti = proto_tree_add_text(tree, tvb, offset, len, "%s Header ", optname);

	dstopt_tree = proto_item_add_subtree(ti, ett_ipv6);

	proto_tree_add_text(dstopt_tree, tvb,
	    offset + offsetof(struct ip6_ext, ip6e_nxt), 1,
	    "Next header: %s (0x%02x)", ipprotostr(ext.ip6e_nxt), ext.ip6e_nxt);
	proto_tree_add_text(dstopt_tree, tvb,
	    offset + offsetof(struct ip6_ext, ip6e_len), 1,
	    "Length: %u (%d bytes)", ext.ip6e_len, len);

	mip_offset = offset;
	mip_offset += 2;

	p = offset + 2;

	while (p < offset + len) {
	    switch (tvb_get_guint8(tvb, p)) {
	    case IP6OPT_PAD1:
		proto_tree_add_text(dstopt_tree, tvb, p, 1, "Pad1");
		p++;
		mip_offset++;
		break;
	    case IP6OPT_PADN:
		tmp = tvb_get_guint8(tvb, p + 1);
		proto_tree_add_text(dstopt_tree, tvb, p, tmp + 2,
		    "PadN: %u bytes", tmp + 2);
		p += tmp;
		p += 2;
		mip_offset += tvb_get_guint8(tvb, mip_offset + 1) + 2;
		break;
	    case IP6OPT_JUMBO:
		tmp = tvb_get_guint8(tvb, p + 1);
		if (tmp == 4) {
		    proto_tree_add_text(dstopt_tree, tvb, p, tmp + 2,
			"Jumbo payload: %u (%u bytes)",
			tvb_get_ntohl(tvb, p + 2), tmp + 2);
		} else {
		    proto_tree_add_text(dstopt_tree, tvb, p, tmp + 2,
			"Jumbo payload: Invalid length (%u bytes)",
			tmp + 2);
		}
		p += tmp;
		p += 2;
		mip_offset += tvb_get_guint8(tvb, mip_offset+1)+2;
		break;
	    case IP6OPT_RTALERT:
	      {
		const char *rta;

		tmp = tvb_get_guint8(tvb, p + 1);
		if (tmp == 2) {
		    rta = val_to_str(tvb_get_ntohs(tvb, p + 2), rtalertvals,
			"Unknown");
		} else
		    rta = "Invalid length";
		ti = proto_tree_add_text(dstopt_tree, tvb, p , tmp + 2,
		    "Router alert: %s (%u bytes)", rta, tmp + 2);
		p += tmp;
		p += 2;
		mip_offset += tvb_get_guint8(tvb, mip_offset + 1) + 2;
		break;
	      }
	    case IP6OPT_HOME_ADDRESS:
		delta = dissect_mipv6_hoa(tvb, dstopt_tree, mip_offset);
		p += delta;
		mip_offset += delta;
		break;
	    default:
		p = offset + len;
		break;
	    }
	}

	/* decode... */
    }
    return len;
}

static int
dissect_hopopts(tvbuff_t *tvb, int offset, proto_tree *tree)
{
    return dissect_opts(tvb, offset, tree, "Hop-by-hop Option");
}

static int
dissect_dstopts(tvbuff_t *tvb, int offset, proto_tree *tree)
{
    return dissect_opts(tvb, offset, tree, "Destination Option");
}

/* START SHIM6 PART */
static guint16 shim_checksum(const guint8 *ptr, int len)
{
	vec_t cksum_vec[1];

	cksum_vec[0].ptr = ptr;
	cksum_vec[0].len = len;
	return in_cksum(&cksum_vec[0], 1);
}

static int 
dissect_shim_ip_addr(tvbuff_t *tvb, int offset, proto_tree *tree, const char *itemname, ...) 
{
    int count;
    struct e_in6_addr addr;

    for(count=0;count<16;count++) {
	addr.bytes[count] = tvb_get_guint8(tvb, offset+count); 
    }

    proto_tree_add_text(tree, tvb, offset, 16, "%s %s", 
	itemname, ip6_to_str(&addr));

    return 16;
}

static int 
dissect_shim_hex(tvbuff_t *tvb, int offset, int len, const char *itemname, guint8 bitmask, proto_tree *tree) 
{
    proto_item *ti;
    int count;
    gint p;

    p = offset;

    ti = proto_tree_add_text(tree, tvb, offset, len, itemname); 

    for (count=0; count<len; count++) 
    {
	if (count==0) {
	    proto_item_append_text(ti, " 0x%02x", tvb_get_guint8(tvb, p+count) & bitmask);
	}
	else {
	    proto_item_append_text(ti, "%02x", tvb_get_guint8(tvb, p+count));
	}
    }

    return len;
}

static const value_string shimoptvals[] = {
    { SHIM6_OPT_RESPVAL, "Responder Validator" },
    { SHIM6_OPT_LOCLIST, "Locator List" },
    { SHIM6_OPT_LOCPREF, "Locator Preferences" },
    { SHIM6_OPT_CGAPDM, "CGA Parameter Data Structure" },
    { SHIM6_OPT_CGASIG, "CGA Signature" },
    { SHIM6_OPT_ULIDPAIR, "ULID Pair" },
    { SHIM6_OPT_FII, "Forked Instance Identifier" },
    { 0, NULL },
};

static const value_string shimverifmethods[] = {
    { SHIM6_VERIF_HBA, "HBA" },
    { SHIM6_VERIF_CGA, "CGA" },
    { 0, NULL },
};

static const value_string shimflags[] = {
    { SHIM6_FLAG_BROKEN, "BROKEN" },
    { SHIM6_FLAG_TEMPORARY, "TEMPORARY" },
    { 0, NULL },
};

static const value_string shimreapstates[] = {
    { SHIM6_REAP_OPERATIONAL, "Operational" },
    { SHIM6_REAP_EXPLORING, "Exploring" },
    { SHIM6_REAP_INBOUNDOK, "InboundOK" },
    { 0, NULL },
};

static int
dissect_shimopts(tvbuff_t *tvb, int offset, proto_tree *tree)
{
    int len, total_len;
    gint p;
    proto_tree *opt_tree;
    proto_item *ti;
    guint8 tmp[4]; 

    p = offset;

    tmp[0] = tvb_get_guint8(tvb, p++);
    tmp[1] = tvb_get_guint8(tvb, p++);
    tmp[2] = tvb_get_guint8(tvb, p++);
    tmp[3] = tvb_get_guint8(tvb, p++);

    len = tmp[2]*256 + tmp[3];
    total_len = (11+len) - ((len+3)%8);

    if (tree) 
    {
	char *ctype;
	char locnum[6];
	int count;
	guint8 optlen;

	/* Option Type */
	ctype = val_to_str( (tvb_get_ntohs(tvb, offset) & SHIM6_BITMASK_OPT_TYPE) >> 1, 
	    shimoptvals, "Unknown Option Type");
	ti = proto_tree_add_text(tree, tvb, offset, total_len, "%s Option", ctype);
	opt_tree = proto_item_add_subtree(ti, ett_ipv6);

    	proto_tree_add_uint_format(opt_tree, hf_ipv6_shim6_opt_type, tvb,
	    offset, 2, (tvb_get_ntohs(tvb, offset) & SHIM6_BITMASK_OPT_TYPE) >> 1, 
	    "Option Type: %s", ctype); 

	/* Critical */
	proto_tree_add_boolean_format(opt_tree, hf_ipv6_shim6_opt_critical, tvb, 
	    offset+1, 1, tmp[1] & SHIM6_BITMASK_CRITICAL, "Critical: %s",
            tmp[1] & SHIM6_BITMASK_CRITICAL ? "Yes" : "No"); 

	/* Content Length */
	proto_tree_add_uint_format(opt_tree, hf_ipv6_shim6_opt_len, tvb, offset+2, 2,
	    len, "Content Length: %u (Total Length: %u)", len, total_len); 

	/* Option Type Specific */
	switch (tvb_get_ntohs(tvb, offset) >> 1) 
	{
	    case SHIM6_OPT_RESPVAL:
		p += dissect_shim_hex(tvb, p, len, "Validator:", 0xff, opt_tree);

		if (total_len-(len+4) > 0)
		    proto_tree_add_text(opt_tree, tvb, p, total_len-(len+4), "Padding");

		break;
	    case SHIM6_OPT_LOCLIST:
	 	proto_tree_add_uint_format(opt_tree, hf_ipv6_shim6_opt_loclist, tvb, p, 4,
		    tvb_get_ntohl(tvb, p), "Locator List Generation: %u (0x%02x)", 
		    tvb_get_ntohl(tvb, p), tvb_get_ntohl(tvb, p));
		p += 4;

		optlen = tvb_get_guint8(tvb, p);
	 	proto_tree_add_uint_format(opt_tree, hf_ipv6_shim6_opt_locnum, tvb, p, 1,
		    optlen, "Num Locators: %u", optlen);
		p++;

		/* Verification Methods */
		proto_tree_add_text(opt_tree, tvb, p, optlen, "Locator Verification Methods:");

		for (count=0;count<optlen;count++) {
		    proto_tree_add_text(opt_tree, tvb, p+count, 1, "(%u): %s", count+1, 
			val_to_str(tvb_get_guint8(tvb, p+count), shimverifmethods, "Unknown"));
		}
		p += optlen;

		/* Padding, included in length field */
		if ((7-optlen % 8) > 0) {
		    proto_tree_add_text(opt_tree, tvb, p, (7-optlen % 8), "Padding");
		    p += (7-optlen % 8);
		}

		/* Locators */
		proto_tree_add_text(opt_tree, tvb, p, 16*optlen, "Locators");

		for (count=0;count<optlen;count++) {
		    g_snprintf(locnum, sizeof locnum, "(%u)", count+1);
		    p += dissect_shim_ip_addr(tvb, p, opt_tree, locnum); 
		}

		break;
	    case SHIM6_OPT_LOCPREF:
	 	proto_tree_add_uint_format(opt_tree, hf_ipv6_shim6_opt_loclist, tvb, p, 4,
		    tvb_get_ntohl(tvb, p), "Locator List Generation: %u (0x%02x)", 
		    tvb_get_ntohl(tvb, p), tvb_get_ntohl(tvb, p));
		p += 4;

		optlen = tvb_get_guint8(tvb, p);
	 	proto_tree_add_uint_format(opt_tree, hf_ipv6_shim6_opt_elemlen, tvb, p, 1,
		    optlen, "Element Lenght: %u", optlen);
		p++;

		/* Locator Preferences */
		count = 0;
		while (p < offset+len+4) {
		    count++;
		    if (optlen > 0) {
			/* Flags */
			proto_tree_add_text(opt_tree, tvb, p, 1, "Locator Flag %u: %s", count, 
			    val_to_str(tvb_get_guint8(tvb, p), shimflags, "Unknown")); 
			p++;

			if (optlen > 1) {
			    /* Priority */
			    proto_tree_add_text(opt_tree, tvb, p, 1, "Locator Priority %u: %u", 
				count, tvb_get_guint8(tvb, p)); 
			    p++;

			    if (optlen > 2) {
				/*
	 			 * This document doesn't specify the format when the Element length is
				 * more than three, except that any such formats MUST be defined so that
				 * the first three octets are the same as in the above case, that is, a
				 * of a 1 octet flags field followed by a 1 octet priority field, and a
				 * 1 octet weight field.
				 */

				/* Weight */
				proto_tree_add_text(opt_tree, tvb, p, 1, "Locator Weight %u: %u", 
				    count, tvb_get_guint8(tvb, p)); 
			   	p++;
			    }
			}
		    }
		}

		if (total_len-(len+4) > 0)
		    proto_tree_add_text(opt_tree, tvb, p, total_len-(len+4), "Padding");

		break;
	    case SHIM6_OPT_CGAPDM:
		p += dissect_shim_hex(tvb, p, len, "CGA Parameter Data Structure:", 0xff, opt_tree); 

		if (total_len-(len+4) > 0)
		    proto_tree_add_text(opt_tree, tvb, p, total_len-(len+4), "Padding");

		break;
	    case SHIM6_OPT_CGASIG:
		p += dissect_shim_hex(tvb, p, len, "CGA Signature:", 0xff, opt_tree); 

		if (total_len-(len+4) > 0)
		    proto_tree_add_text(opt_tree, tvb, p, total_len-(len+4), "Padding");

		break;
	    case SHIM6_OPT_ULIDPAIR:
		proto_tree_add_text(opt_tree, tvb, p, 4, "Reserved");
		p += 4;

		p += dissect_shim_ip_addr(tvb, p, opt_tree, "Sender ULID:"); 
		p += dissect_shim_ip_addr(tvb, p, opt_tree, "Receiver ULID:"); 

		break;
	    case SHIM6_OPT_FII:
	 	proto_tree_add_uint_format(opt_tree, hf_ipv6_shim6_opt_fii, tvb, p, 4,
		    tvb_get_ntohl(tvb, p), "Forked Instance Identifier: %u (0x%02x)", 
		    tvb_get_ntohl(tvb, p), tvb_get_ntohl(tvb, p));
		p += 4;
		break;
	    default:
		break;
	}
    }

    return total_len;
}

/* Dissect SHIM6 data: control messages */
static int
dissect_shimctrl(tvbuff_t *tvb, int offset, guint type, proto_tree *shim_tree)
{
    gint p;
    guint8 tmp[6];
    const char *sta;
    int count;

    p = offset; 

    switch (type) 
    {
	case SHIM6_TYPE_I1:
	    tmp[0] = tvb_get_guint8(tvb, p++);
	    tmp[1] = tvb_get_guint8(tvb, p++);
	    tmp[2] = tvb_get_guint8(tvb, p++);
	    tmp[3] = tvb_get_guint8(tvb, p++);
	    tmp[4] = tvb_get_guint8(tvb, p++);
	    tmp[5] = tvb_get_guint8(tvb, p++);

	    proto_tree_add_none_format(shim_tree, hf_ipv6_shim6_ct, tvb,
		p-6, 6, "Initiator Context Tag: %02x %02x %02x %02x %02x %02x", 
		tmp[0] & SHIM6_BITMASK_CT, tmp[1], tmp[2], tmp[3], tmp[4], tmp[5]); 

	    proto_tree_add_uint_format(shim_tree, hf_ipv6_shim6_inonce, tvb, p, 4,
		    tvb_get_ntohl(tvb, p), "Initiator Nonce: %u (0x%02x)", 
		    tvb_get_ntohl(tvb, p), tvb_get_ntohl(tvb, p));
	    p += 4;

	    break;
	case SHIM6_TYPE_R1:
	    proto_tree_add_text(shim_tree, tvb, p, 2, "Reserved2");
	    p += 2;

	    proto_tree_add_uint_format(shim_tree, hf_ipv6_shim6_inonce, tvb, p, 4,
		    tvb_get_ntohl(tvb, p), "Initiator Nonce: %u (0x%02x)", 
		    tvb_get_ntohl(tvb, p), tvb_get_ntohl(tvb, p));
	    p += 4;

	    proto_tree_add_uint_format(shim_tree, hf_ipv6_shim6_rnonce, tvb, p, 4,
		    tvb_get_ntohl(tvb, p), "Responder Nonce: %u (0x%02x)", 
		    tvb_get_ntohl(tvb, p), tvb_get_ntohl(tvb, p));
	    p += 4;

	    break;
	case SHIM6_TYPE_I2:
	    tmp[0] = tvb_get_guint8(tvb, p++);
	    tmp[1] = tvb_get_guint8(tvb, p++);
	    tmp[2] = tvb_get_guint8(tvb, p++);
	    tmp[3] = tvb_get_guint8(tvb, p++);
	    tmp[4] = tvb_get_guint8(tvb, p++);
	    tmp[5] = tvb_get_guint8(tvb, p++);

	    proto_tree_add_none_format(shim_tree, hf_ipv6_shim6_ct, tvb,
		p-6, 6, "Initiator Context Tag: %02x %02x %02x %02x %02x %02x", 
		tmp[0] & SHIM6_BITMASK_CT, tmp[1], tmp[2], tmp[3], tmp[4], tmp[5]); 

	    proto_tree_add_uint_format(shim_tree, hf_ipv6_shim6_inonce, tvb, p, 4,
		    tvb_get_ntohl(tvb, p), "Initiator Nonce: %u (0x%02x)", 
		    tvb_get_ntohl(tvb, p), tvb_get_ntohl(tvb, p));
	    p += 4;

	    proto_tree_add_uint_format(shim_tree, hf_ipv6_shim6_rnonce, tvb, p, 4,
		    tvb_get_ntohl(tvb, p), "Responder Nonce: %u (0x%02x)", 
		    tvb_get_ntohl(tvb, p), tvb_get_ntohl(tvb, p));
	    p += 4;

	    proto_tree_add_text(shim_tree, tvb, p, 4, "Reserved2");
	    p += 4;

	    break;
	case SHIM6_TYPE_R2:
	    tmp[0] = tvb_get_guint8(tvb, p++);
	    tmp[1] = tvb_get_guint8(tvb, p++);
	    tmp[2] = tvb_get_guint8(tvb, p++);
	    tmp[3] = tvb_get_guint8(tvb, p++);
	    tmp[4] = tvb_get_guint8(tvb, p++);
	    tmp[5] = tvb_get_guint8(tvb, p++);

	    proto_tree_add_none_format(shim_tree, hf_ipv6_shim6_ct, tvb,
		p-6, 6, "Responder Context Tag: %02x %02x %02x %02x %02x %02x", 
		tmp[0] & SHIM6_BITMASK_CT, tmp[1], tmp[2], tmp[3], tmp[4], tmp[5]); 

	    proto_tree_add_uint_format(shim_tree, hf_ipv6_shim6_inonce, tvb, p, 4,
		    tvb_get_ntohl(tvb, p), "Initiator Nonce: %u (0x%02x)", 
		    tvb_get_ntohl(tvb, p), tvb_get_ntohl(tvb, p));
	    p += 4;

	    break;
	case SHIM6_TYPE_R1BIS:
	    tmp[0] = tvb_get_guint8(tvb, p++);
	    tmp[1] = tvb_get_guint8(tvb, p++);
	    tmp[2] = tvb_get_guint8(tvb, p++);
	    tmp[3] = tvb_get_guint8(tvb, p++);
	    tmp[4] = tvb_get_guint8(tvb, p++);
	    tmp[5] = tvb_get_guint8(tvb, p++);

	    proto_tree_add_none_format(shim_tree, hf_ipv6_shim6_ct, tvb,
		p-6, 6, "Packet Context Tag: %02x %02x %02x %02x %02x %02x", 
		tmp[0] & SHIM6_BITMASK_CT, tmp[1], tmp[2], tmp[3], tmp[4], tmp[5]); 

	    proto_tree_add_uint_format(shim_tree, hf_ipv6_shim6_rnonce, tvb, p, 4,
		    tvb_get_ntohl(tvb, p), "Responder Nonce: %u (0x%02x)", 
		    tvb_get_ntohl(tvb, p), tvb_get_ntohl(tvb, p));
	    p += 4;

	    break;
	case SHIM6_TYPE_I2BIS:
	    tmp[0] = tvb_get_guint8(tvb, p++);
	    tmp[1] = tvb_get_guint8(tvb, p++);
	    tmp[2] = tvb_get_guint8(tvb, p++);
	    tmp[3] = tvb_get_guint8(tvb, p++);
	    tmp[4] = tvb_get_guint8(tvb, p++);
	    tmp[5] = tvb_get_guint8(tvb, p++);

	    proto_tree_add_none_format(shim_tree, hf_ipv6_shim6_ct, tvb,
		p-6, 6, "Initiator Context Tag: %02x %02x %02x %02x %02x %02x", 
		tmp[0] & SHIM6_BITMASK_CT, tmp[1], tmp[2], tmp[3], tmp[4], tmp[5]); 

	    proto_tree_add_uint_format(shim_tree, hf_ipv6_shim6_inonce, tvb, p, 4,
		    tvb_get_ntohl(tvb, p), "Initiator Nonce: %u (0x%02x)", 
		    tvb_get_ntohl(tvb, p), tvb_get_ntohl(tvb, p));
	    p += 4;

	    proto_tree_add_uint_format(shim_tree, hf_ipv6_shim6_rnonce, tvb, p, 4,
		    tvb_get_ntohl(tvb, p), "Responder Nonce: %u (0x%02x)", 
		    tvb_get_ntohl(tvb, p), tvb_get_ntohl(tvb, p));
	    p += 4;

	    proto_tree_add_text(shim_tree, tvb, p, 6, "Reserved2");
	    p += 6;
	    
	    tmp[0] = tvb_get_guint8(tvb, p++);
	    tmp[1] = tvb_get_guint8(tvb, p++);
	    tmp[2] = tvb_get_guint8(tvb, p++);
	    tmp[3] = tvb_get_guint8(tvb, p++);
	    tmp[4] = tvb_get_guint8(tvb, p++);
	    tmp[5] = tvb_get_guint8(tvb, p++);

	    proto_tree_add_none_format(shim_tree, hf_ipv6_shim6_ct, tvb,
		p-6, 6, "Initiator Context Tag: %02x %02x %02x %02x %02x %02x", 
		tmp[0] & SHIM6_BITMASK_CT, tmp[1], tmp[2], tmp[3], tmp[4], tmp[5]); 

	    break;
	case SHIM6_TYPE_UPD_REQ:
	case SHIM6_TYPE_UPD_ACK:
	    tmp[0] = tvb_get_guint8(tvb, p++);
	    tmp[1] = tvb_get_guint8(tvb, p++);
	    tmp[2] = tvb_get_guint8(tvb, p++);
	    tmp[3] = tvb_get_guint8(tvb, p++);
	    tmp[4] = tvb_get_guint8(tvb, p++);
	    tmp[5] = tvb_get_guint8(tvb, p++);

	    proto_tree_add_none_format(shim_tree, hf_ipv6_shim6_ct, tvb,
		p-6, 6, "Receiver Context Tag: %02x %02x %02x %02x %02x %02x", 
		tmp[0] & SHIM6_BITMASK_CT, tmp[1], tmp[2], tmp[3], tmp[4], tmp[5]); 

	    proto_tree_add_uint_format(shim_tree, hf_ipv6_shim6_inonce, tvb, p, 4,
		    tvb_get_ntohl(tvb, p), "Request Nonce: %u (0x%02x)", 
		    tvb_get_ntohl(tvb, p), tvb_get_ntohl(tvb, p));
	    p += 4;

	    break;
	case SHIM6_TYPE_KEEPALIVE:
	    tmp[0] = tvb_get_guint8(tvb, p++);
	    tmp[1] = tvb_get_guint8(tvb, p++);
	    tmp[2] = tvb_get_guint8(tvb, p++);
	    tmp[3] = tvb_get_guint8(tvb, p++);
	    tmp[4] = tvb_get_guint8(tvb, p++);
	    tmp[5] = tvb_get_guint8(tvb, p++);

	    proto_tree_add_none_format(shim_tree, hf_ipv6_shim6_ct, tvb,
		p-6, 6, "Receiver Context Tag: %02x %02x %02x %02x %02x %02x", 
		tmp[0] & SHIM6_BITMASK_CT, tmp[1], tmp[2], tmp[3], tmp[4], tmp[5]); 

	    proto_tree_add_text(shim_tree, tvb, p, 4, "Reserved2");
	    p += 4;
	    break;
	case SHIM6_TYPE_PROBE:
	    tmp[0] = tvb_get_guint8(tvb, p++);
	    tmp[1] = tvb_get_guint8(tvb, p++);
	    tmp[2] = tvb_get_guint8(tvb, p++);
	    tmp[3] = tvb_get_guint8(tvb, p++);
	    tmp[4] = tvb_get_guint8(tvb, p++);
	    tmp[5] = tvb_get_guint8(tvb, p++);

	    proto_tree_add_none_format(shim_tree, hf_ipv6_shim6_ct, tvb,
		p-6, 6, "Receiver Context Tag: %02x %02x %02x %02x %02x %02x", 
		tmp[0] & SHIM6_BITMASK_CT, tmp[1], tmp[2], tmp[3], tmp[4], tmp[5]); 

	    tmp[0] = tvb_get_guint8(tvb, p);
	    proto_tree_add_uint_format(shim_tree, hf_ipv6_shim6_precvd, tvb,
	    	p, 1, (tmp[0] & SHIM6_BITMASK_PRECVD) >> 4,
		"Probes Received: %u", (tmp[0] & SHIM6_BITMASK_PRECVD) >> 4); 
	    proto_tree_add_uint_format(shim_tree, hf_ipv6_shim6_psent, tvb,
	    	p, 1, tmp[0] & SHIM6_BITMASK_PSENT,
		"Probes Sent: %u", tmp[0] & SHIM6_BITMASK_PSENT); 

	    p++;

	    sta = val_to_str((tvb_get_guint8(tvb, p) & SHIM6_BITMASK_STA) >> 6,
		shimreapstates, "Unknown REAP State");

	    proto_tree_add_uint_format(shim_tree, hf_ipv6_shim6_reap, tvb,
	    	p, 1, (tvb_get_guint8(tvb, p) & SHIM6_BITMASK_STA) >> 6,
		"REAP State: %s", sta); 

	    proto_tree_add_text(shim_tree, tvb, p, 3, "Reserved2");
	    p += 3;
	    
	    /* Probes Sent */
	    for (count=0;count<(tmp[0] & SHIM6_BITMASK_PSENT);count++) {
		proto_tree_add_text(shim_tree, tvb, p, 40, "Probe Sent %u:", count+1);

		p += dissect_shim_ip_addr(tvb, p, shim_tree, "- Source address:"); 
		p += dissect_shim_ip_addr(tvb, p, shim_tree, "- Destination address:"); 

		proto_tree_add_text(shim_tree, tvb, p, 4, "- Nonce: %u (0x%02x)", 
		    tvb_get_ntohl(tvb, p), tvb_get_ntohl(tvb, p));
		p += 4;

		proto_tree_add_text(shim_tree, tvb, p, 4, "- Data: 0x%02x", 
		    tvb_get_ntohl(tvb, p));
		p += 4;
	   }

	   /* Probes Received */
	   for (count=0;count<((tmp[0] & SHIM6_BITMASK_PRECVD)>>4);count++) {
		proto_tree_add_text(shim_tree, tvb, p, 40, "Probe Received %u:", count+1);

		p += dissect_shim_ip_addr(tvb, p, shim_tree, "- Source address:"); 
		p += dissect_shim_ip_addr(tvb, p, shim_tree, "- Destination address:"); 

		proto_tree_add_text(shim_tree, tvb, p, 4, "- Nonce: %u (0x%02x)", 
		    tvb_get_ntohl(tvb, p), tvb_get_ntohl(tvb, p));
		p += 4;

		proto_tree_add_text(shim_tree, tvb, p, 4, "- Data: 0x%02x", 
		    tvb_get_ntohl(tvb, p));
		p += 4;
	   }

	   break;
	default:
	   break;
    }

    return p-offset;
}

/* Dissect SHIM6 data: payload, common part, options */
static const value_string shimctrlvals[] = {
    { SHIM6_TYPE_I1, "I1" },
    { SHIM6_TYPE_R1, "R1" },
    { SHIM6_TYPE_I2, "I2" },
    { SHIM6_TYPE_R2, "R2" },
    { SHIM6_TYPE_R1BIS, "R1bis" },
    { SHIM6_TYPE_I2BIS, "I2bis" },
    { SHIM6_TYPE_UPD_REQ, "Update Request" },
    { SHIM6_TYPE_UPD_ACK, "Update Acknowledgement" },
    { SHIM6_TYPE_KEEPALIVE, "Keepalive" },
    { SHIM6_TYPE_PROBE, "Probe" },
    { 0, NULL },
};

static int
dissect_shim6(tvbuff_t *tvb, int offset, proto_tree *tree) 
{
    struct ip6_shim shim;		
    int len;
    gint p;
    proto_tree *shim_tree, *opt_tree;
    proto_item *ti;
    guint8 tmp[5]; 

    tvb_memcpy(tvb, (guint8 *)&shim, offset, sizeof(shim));
    len = (shim.ip6s_len + 1) << 3;

    if (tree) 
    {
    	ti = proto_tree_add_item(tree, hf_ipv6_shim6, tvb, offset, len, FALSE);
	shim_tree = proto_item_add_subtree(ti, ett_ipv6);

	/* Next Header */
	proto_tree_add_uint_format(shim_tree, hf_ipv6_shim6_nxt, tvb,
	    offset + offsetof(struct ip6_shim, ip6s_nxt), 1, shim.ip6s_nxt,
	    "Next header: %s (0x%02x)", ipprotostr(shim.ip6s_nxt), shim.ip6s_nxt); 

	/* Header Extension Length */
	proto_tree_add_uint_format(shim_tree, hf_ipv6_shim6_len, tvb,
	    offset + offsetof(struct ip6_shim, ip6s_len), 1, shim.ip6s_len,
	    "Header Ext Length: %u (%d bytes)", shim.ip6s_len, len); 

	/* P Field */
	proto_tree_add_boolean(shim_tree, hf_ipv6_shim6_p, tvb, 
	    offset + offsetof(struct ip6_shim, ip6s_p), 1, shim.ip6s_p & SHIM6_BITMASK_P);

	/* skip the first 2 bytes (nxt hdr, hdr ext len, p+7bits) */
	p = offset + 3; 

	if (shim.ip6s_p & SHIM6_BITMASK_P) 
	{ 
	    tmp[0] = tvb_get_guint8(tvb, p++);
	    tmp[1] = tvb_get_guint8(tvb, p++);
	    tmp[2] = tvb_get_guint8(tvb, p++);
	    tmp[3] = tvb_get_guint8(tvb, p++);
	    tmp[4] = tvb_get_guint8(tvb, p++);

	    /* Payload Extension Header */
	    proto_tree_add_none_format(shim_tree, hf_ipv6_shim6_ct, tvb,
		offset + offsetof(struct ip6_shim, ip6s_p), 6, 
		"Receiver Context Tag: %02x %02x %02x %02x %02x %02x", 
		shim.ip6s_p & SHIM6_BITMASK_CT, tmp[0], tmp[1], tmp[2], tmp[3], tmp[4]); 
	}
	else 
        { 
	    /* Control Message */
	    const char *ctype;
	    proto_item *ti_shim;
	    guint16 csum;
	    int advance;

	    /* Message Type */
	    ctype = val_to_str(shim.ip6s_p & SHIM6_BITMASK_TYPE, shimctrlvals,
		"Unknown Message Type");

	    proto_tree_add_uint_format(shim_tree, hf_ipv6_shim6_type, tvb,
	    offset + offsetof(struct ip6_shim, ip6s_p), 1, shim.ip6s_p & SHIM6_BITMASK_TYPE,
	    "Message Type: %s", ctype); 

	    /* Protocol bit (Must be zero for SHIM6) */
	    proto_tree_add_boolean_format(shim_tree, hf_ipv6_shim6_proto, tvb, p, 1, 
		tvb_get_guint8(tvb, p) & SHIM6_BITMASK_PROTOCOL, "Protocol: %s", 
		tvb_get_guint8(tvb, p) & SHIM6_BITMASK_PROTOCOL ? "HIP" : "SHIM6");
	    p++;

	    /* Checksum */ 
	    csum = shim_checksum(tvb_get_ptr(tvb, offset, len), len);

	    if (csum == 0) {
		proto_tree_add_uint_format(shim_tree, hf_ipv6_shim6_csum, tvb, p, 2,
		    tvb_get_ntohs(tvb, p), "Checksum: 0x%04x [correct]", tvb_get_ntohs(tvb, p));
	    } else {
		proto_tree_add_uint_format(shim_tree, hf_ipv6_shim6_csum, tvb, p, 2,
		    tvb_get_ntohs(tvb, p), "Checksum: 0x%04x [incorrect: should be 0x%04x]", 
		    tvb_get_ntohs(tvb, p), in_cksum_shouldbe(tvb_get_ntohs(tvb, p), csum));
	    }
	    p += 2;

	    /* Type specific data */
	    advance = dissect_shimctrl(tvb, p, shim.ip6s_p & SHIM6_BITMASK_TYPE, shim_tree);
		
	    p += advance;

	    /* Options */

	    if (p < offset+len && shim_tree) {
		ti_shim = proto_tree_add_text(shim_tree, tvb, p, len-(advance+6), 
		    "Options (%u bytes)", len-(advance+6));
		
		opt_tree = proto_item_add_subtree(ti_shim, ett_ipv6);

		while (p < offset+len) {
		    p += dissect_shimopts(tvb, p, opt_tree);
		}
	    }

	}
    }

    return len;
}

/* END SHIM6 PART */

static void
dissect_ipv6(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree)
{
  proto_tree *ipv6_tree = NULL;
  proto_item *ti;
  guint8 nxt, stype;
  int advance;
  int poffset;
  guint16 plen;
  gboolean hopopts, routing, frag, ah, shim6, dstopts;
  guint16 offlg;
  guint32 ident;
  int offset;
  fragment_data *ipfd_head;
  tvbuff_t   *next_tvb;
  gboolean update_col_info = TRUE;
  gboolean save_fragmented;

  struct ip6_hdr ipv6;

  if (check_col(pinfo->cinfo, COL_PROTOCOL))
    col_set_str(pinfo->cinfo, COL_PROTOCOL, "IPv6");
  if (check_col(pinfo->cinfo, COL_INFO))
    col_clear(pinfo->cinfo, COL_INFO);

  offset = 0;
  tvb_memcpy(tvb, (guint8 *)&ipv6, offset, sizeof(ipv6));

  /* Get extension header and payload length */
  plen = g_ntohs(ipv6.ip6_plen);

  /* Adjust the length of this tvbuff to include only the IPv6 datagram. */
  set_actual_length(tvb, plen + sizeof (struct ip6_hdr));

  SET_ADDRESS(&pinfo->net_src, AT_IPv6, 16, tvb_get_ptr(tvb, offset + IP6H_SRC, 16));
  SET_ADDRESS(&pinfo->src, AT_IPv6, 16, tvb_get_ptr(tvb, offset + IP6H_SRC, 16));
  SET_ADDRESS(&pinfo->net_dst, AT_IPv6, 16, tvb_get_ptr(tvb, offset + IP6H_DST, 16));
  SET_ADDRESS(&pinfo->dst, AT_IPv6, 16, tvb_get_ptr(tvb, offset + IP6H_DST, 16));

  if (tree) {
    /* !!! specify length */
    ti = proto_tree_add_item(tree, proto_ipv6, tvb, offset, 40, FALSE);
    ipv6_tree = proto_item_add_subtree(ti, ett_ipv6);

    /* !!! warning: version also contains 4 Bit priority */
    proto_tree_add_uint(ipv6_tree, hf_ipv6_version, tvb,
		offset + offsetof(struct ip6_hdr, ip6_vfc), 1,
		(ipv6.ip6_vfc >> 4) & 0x0f);

    proto_tree_add_uint(ipv6_tree, hf_ipv6_class, tvb,
		offset + offsetof(struct ip6_hdr, ip6_flow), 4,
		(guint8)((g_ntohl(ipv6.ip6_flow) >> 20) & 0xff));

    /*
     * there should be no alignment problems for ip6_flow, since it's the first
     * guint32 in the ipv6 struct
     */
    proto_tree_add_uint_format(ipv6_tree, hf_ipv6_flow, tvb,
		offset + offsetof(struct ip6_hdr, ip6_flow), 4,
		(unsigned long)(g_ntohl(ipv6.ip6_flow) & IPV6_FLOWLABEL_MASK),
		"Flowlabel: 0x%05lx",
		(unsigned long)(g_ntohl(ipv6.ip6_flow) & IPV6_FLOWLABEL_MASK));

    proto_tree_add_uint(ipv6_tree, hf_ipv6_plen, tvb,
		offset + offsetof(struct ip6_hdr, ip6_plen), 2,
		plen);

    proto_tree_add_uint_format(ipv6_tree, hf_ipv6_nxt, tvb,
		offset + offsetof(struct ip6_hdr, ip6_nxt), 1,
		ipv6.ip6_nxt,
		"Next header: %s (0x%02x)",
		ipprotostr(ipv6.ip6_nxt), ipv6.ip6_nxt);

    proto_tree_add_uint(ipv6_tree, hf_ipv6_hlim, tvb,
		offset + offsetof(struct ip6_hdr, ip6_hlim), 1,
		ipv6.ip6_hlim);

    proto_tree_add_ipv6_hidden(ipv6_tree, hf_ipv6_addr, tvb,
			       offset + offsetof(struct ip6_hdr, ip6_src), 16,
			       ipv6.ip6_src.bytes);
    proto_tree_add_ipv6_hidden(ipv6_tree, hf_ipv6_addr, tvb,
			       offset + offsetof(struct ip6_hdr, ip6_dst), 16,
			       ipv6.ip6_dst.bytes);

    proto_tree_add_ipv6_format(ipv6_tree, hf_ipv6_src, tvb,
		offset + offsetof(struct ip6_hdr, ip6_src), 16,
		(guint8 *)&ipv6.ip6_src,
#ifdef INET6
		"Source address: %s (%s)",
		get_hostname6(&ipv6.ip6_src),
#else
		"Source address: %s",
#endif
		ip6_to_str(&ipv6.ip6_src));

    proto_tree_add_ipv6_format(ipv6_tree, hf_ipv6_dst, tvb,
		offset + offsetof(struct ip6_hdr, ip6_dst), 16,
		(guint8 *)&ipv6.ip6_dst,
#ifdef INET6
		"Destination address: %s (%s)",
		get_hostname6(&ipv6.ip6_dst),
#else
		"Destination address: %s",
#endif
		ip6_to_str(&ipv6.ip6_dst));
  }

  /* start of the new header (could be a extension header) */
  poffset = offset + offsetof(struct ip6_hdr, ip6_nxt);
  nxt = tvb_get_guint8(tvb, poffset);
  offset += sizeof(struct ip6_hdr);
  offlg = 0;
  ident = 0;

/* start out assuming this isn't fragmented, and has none of the other
   non-final headers */
  hopopts = FALSE;
  routing = FALSE;
  frag = FALSE;
  ah = FALSE;
  shim6 = FALSE;
  dstopts = FALSE;

again:
   switch (nxt) {
   case IP_PROTO_HOPOPTS:
			hopopts = TRUE;
			advance = dissect_hopopts(tvb, offset, tree);
			nxt = tvb_get_guint8(tvb, offset);
			poffset = offset;
			offset += advance;
			plen -= advance;
			goto again;
    case IP_PROTO_ROUTING:
			routing = TRUE;
			advance = dissect_routing6(tvb, offset, tree);
			nxt = tvb_get_guint8(tvb, offset);
			poffset = offset;
			offset += advance;
			plen -= advance;
			goto again;
    case IP_PROTO_FRAGMENT:
			frag = TRUE;
			advance = dissect_frag6(tvb, offset, pinfo, tree,
			    &offlg, &ident);
			nxt = tvb_get_guint8(tvb, offset);
			poffset = offset;
			offset += advance;
			plen -= advance;
			goto again;
    case IP_PROTO_AH:
			ah = TRUE;
			advance = dissect_ah_header(
				  tvb_new_subset(tvb, offset, -1, -1),
				  pinfo, tree, NULL, NULL);
			nxt = tvb_get_guint8(tvb, offset);
			poffset = offset;
			offset += advance;
			plen -= advance;
			goto again;
    case IP_PROTO_SHIM6:
			shim6 = TRUE;
			advance = dissect_shim6(tvb, offset, tree);
			nxt = tvb_get_guint8(tvb, offset);
			stype = tvb_get_guint8(tvb, offset+2); 
			poffset = offset;
			offset += advance;
			plen -= advance;
			goto again;
    case IP_PROTO_DSTOPTS:
			dstopts = TRUE;
			advance = dissect_dstopts(tvb, offset, tree);
			nxt = tvb_get_guint8(tvb, offset);
			poffset = offset;
			offset += advance;
			plen -= advance;
			goto again;
    }

#ifdef TEST_FINALHDR
  proto_tree_add_uint_hidden(ipv6_tree, hf_ipv6_final, tvb, poffset, 1, nxt);
#endif

  /* collect packet info */
  pinfo->ipproto = nxt;
  pinfo->iplen = sizeof(ipv6) + plen + offset;
  pinfo->iphdrlen = offset;

  /* If ipv6_reassemble is on, this is a fragment, and we have all the data
   * in the fragment, then just add the fragment to the hashtable.
   */
  save_fragmented = pinfo->fragmented;
  if (ipv6_reassemble && frag && tvb_bytes_exist(tvb, offset, plen)) {
    ipfd_head = fragment_add_check(tvb, offset, pinfo, ident,
			     ipv6_fragment_table,
			     ipv6_reassembled_table,
			     offlg & IP6F_OFF_MASK,
			     plen,
			     offlg & IP6F_MORE_FRAG);

    next_tvb = process_reassembled_data(tvb, offset, pinfo, "Reassembled IPv6",
          ipfd_head, &ipv6_frag_items, &update_col_info, ipv6_tree);
  } else {
    /* If this is the first fragment, dissect its contents, otherwise
       just show it as a fragment.

       XXX - if we eventually don't save the reassembled contents of all
       fragmented datagrams, we may want to always reassemble. */
    if (offlg & IP6F_OFF_MASK) {
      /* Not the first fragment - don't dissect it. */
      next_tvb = NULL;
    } else {
      /* First fragment, or not fragmented.  Dissect what we have here. */

      /* Get a tvbuff for the payload. */
      next_tvb = tvb_new_subset(tvb, offset, -1, -1);

      /*
       * If this is the first fragment, but not the only fragment,
       * tell the next protocol that.
       */
      if (offlg & IP6F_MORE_FRAG)
        pinfo->fragmented = TRUE;
      else
        pinfo->fragmented = FALSE;
    }
  }

  if (next_tvb == NULL) {
    /* Just show this as a fragment. */
    /* COL_INFO was filled in by "dissect_frag6()" */
    call_dissector(data_handle, tvb_new_subset(tvb, offset, -1, -1), pinfo, tree);

    /* As we haven't reassembled anything, we haven't changed "pi", so
       we don't have to restore it. */
    pinfo->fragmented = save_fragmented;
    return;
  }

  /* do lookup with the subdissector table */
  if (!dissector_try_port(ip_dissector_table, nxt, next_tvb, pinfo, tree)) {
    /* Unknown protocol.
       Handle "no next header" specially. */
    if (nxt == IP_PROTO_NONE) {
      if (check_col(pinfo->cinfo, COL_INFO)) {
        /* If we had an Authentication Header, the AH dissector already
           put something in the Info column; leave it there. */
      	if (!ah) {
          if (hopopts || routing || dstopts || shim6) {
            const char *sep = "IPv6 ";
            if (hopopts) {
              col_append_fstr(pinfo->cinfo, COL_INFO, "%shop-by-hop options",
                             sep);
              sep = ", ";
            }
            if (routing) {
              col_append_fstr(pinfo->cinfo, COL_INFO, "%srouting", sep);
              sep = ", ";
            }
            if (dstopts) {
              col_append_fstr(pinfo->cinfo, COL_INFO, "%sdestination options",
                              sep);
            }
            if (shim6) {
		if (stype & SHIM6_BITMASK_P) {
              	  col_append_fstr(pinfo->cinfo, COL_INFO, "Shim6 (Payload)");
		}
		else {
              	  col_append_fstr(pinfo->cinfo, COL_INFO, "Shim6 (%s)", 
		    val_to_str(stype & SHIM6_BITMASK_TYPE, shimctrlvals, "Unknown"));
		}
            }
          } else
            col_set_str(pinfo->cinfo, COL_INFO, "IPv6 no next header");
	}
      }
    } else {
      if (check_col(pinfo->cinfo, COL_INFO))
        col_add_fstr(pinfo->cinfo, COL_INFO, "%s (0x%02x)", ipprotostr(nxt),nxt);
    }
    call_dissector(data_handle, next_tvb, pinfo, tree);
  }
  pinfo->fragmented = save_fragmented;
}

void
proto_register_ipv6(void)
{
  static hf_register_info hf[] = {
    { &hf_ipv6_version,
      { "Version",		"ipv6.version",
				FT_UINT8, BASE_DEC, NULL, 0x0, "", HFILL }},
    { &hf_ipv6_class,
      { "Traffic class",	"ipv6.class",
				FT_UINT8, BASE_HEX, NULL, 0x0, "", HFILL }},
    { &hf_ipv6_flow,
      { "Flowlabel",		"ipv6.flow",
				FT_UINT32, BASE_HEX, NULL, 0x0, "", HFILL }},
    { &hf_ipv6_plen,
      { "Payload length",	"ipv6.plen",
				FT_UINT16, BASE_DEC, NULL, 0x0, "", HFILL }},
    { &hf_ipv6_nxt,
      { "Next header",		"ipv6.nxt",
				FT_UINT8, BASE_HEX, NULL, 0x0, "", HFILL }},
    { &hf_ipv6_hlim,
      { "Hop limit",		"ipv6.hlim",
				FT_UINT8, BASE_DEC, NULL, 0x0, "", HFILL }},
    { &hf_ipv6_src,
      { "Source",		"ipv6.src",
				FT_IPv6, BASE_NONE, NULL, 0x0,
				"Source IPv6 Address", HFILL }},
    { &hf_ipv6_dst,
      { "Destination",		"ipv6.dst",
				FT_IPv6, BASE_NONE, NULL, 0x0,
				"Destination IPv6 Address", HFILL }},
    { &hf_ipv6_addr,
      { "Address",		"ipv6.addr",
				FT_IPv6, BASE_NONE, NULL, 0x0,
				"Source or Destination IPv6 Address", HFILL }},

    { &hf_ipv6_fragment_overlap,
      { "Fragment overlap",	"ipv6.fragment.overlap",
				FT_BOOLEAN, BASE_NONE, NULL, 0x0,
				"Fragment overlaps with other fragments", HFILL }},

    { &hf_ipv6_fragment_overlap_conflict,
      { "Conflicting data in fragment overlap",	"ipv6.fragment.overlap.conflict",
				FT_BOOLEAN, BASE_NONE, NULL, 0x0,
				"Overlapping fragments contained conflicting data", HFILL }},

    { &hf_ipv6_fragment_multiple_tails,
      { "Multiple tail fragments found", "ipv6.fragment.multipletails",
				FT_BOOLEAN, BASE_NONE, NULL, 0x0,
				"Several tails were found when defragmenting the packet", HFILL }},

    { &hf_ipv6_fragment_too_long_fragment,
      { "Fragment too long",	"ipv6.fragment.toolongfragment",
				FT_BOOLEAN, BASE_NONE, NULL, 0x0,
				"Fragment contained data past end of packet", HFILL }},

    { &hf_ipv6_fragment_error,
      { "Defragmentation error", "ipv6.fragment.error",
				FT_FRAMENUM, BASE_NONE, NULL, 0x0,
				"Defragmentation error due to illegal fragments", HFILL }},

    { &hf_ipv6_fragment,
      { "IPv6 Fragment",	"ipv6.fragment",
				FT_FRAMENUM, BASE_NONE, NULL, 0x0,
				"IPv6 Fragment", HFILL }},

    { &hf_ipv6_fragments,
      { "IPv6 Fragments",	"ipv6.fragments",
				FT_NONE, BASE_NONE, NULL, 0x0,
				"IPv6 Fragments", HFILL }},

    { &hf_ipv6_reassembled_in,
      { "Reassembled IPv6 in frame", "ipv6.reassembled_in",
				FT_FRAMENUM, BASE_NONE, NULL, 0x0,
				"This IPv6 packet is reassembled in this frame", HFILL }},

    /* Mobile IPv6 */
    { &hf_ipv6_mipv6_type,
      { "Option Type ",		"ipv6.mipv6_type",
				FT_UINT8, BASE_DEC, NULL, 0x0,
				"", HFILL }},
    { &hf_ipv6_mipv6_length,
      { "Option Length ",	"ipv6.mipv6_length",
				FT_UINT8, BASE_DEC, NULL, 0x0,
				"", HFILL }},
    { &hf_ipv6_mipv6_home_address,
      { "Home Address ",	"ipv6.mipv6_home_address",
				FT_IPv6, BASE_HEX, NULL, 0x0,
				"", HFILL }},

    /* SHIM6 */
    { &hf_ipv6_shim6,
      { "SHIM6 ",		"ipv6.shim6",
				FT_NONE, BASE_NONE, NULL, 0x0,
				"", HFILL }},

    { &hf_ipv6_shim6_nxt,
      { "Next Header",		"ipv6.shim6.nxt",
				FT_UINT8, BASE_DEC, NULL, 0x0,
				"", HFILL }},

    { &hf_ipv6_shim6_len,
      { "Header Ext Length", 	"ipv6.shim6.len",
				FT_UINT8, BASE_DEC, NULL, 0x0,
				"", HFILL }},

    { &hf_ipv6_shim6_p,
      { "P Bit", 		"ipv6.shim6.p",
				FT_BOOLEAN, BASE_NONE, NULL, 0x0,
				"", HFILL }},

    { &hf_ipv6_shim6_ct,
      { "Context Tag", 		"ipv6.shim6.ct",
				FT_NONE, BASE_NONE, NULL, 0x0,
				"", HFILL }},

    { &hf_ipv6_shim6_type,
      { "Message Type", 	"ipv6.shim6.type",
				FT_UINT8, BASE_DEC, NULL, 0x0,
				"", HFILL }},

    { &hf_ipv6_shim6_proto,
      { "Protocol", 		"ipv6.shim6.proto",
				FT_BOOLEAN, BASE_NONE, NULL, 0x0,
				"", HFILL }},

    { &hf_ipv6_shim6_csum,
      { "Checksum", 		"ipv6.shim6.csum",
				FT_UINT16, BASE_HEX, NULL, 0x0,
				"", HFILL }},

    { &hf_ipv6_shim6_inonce,
      { "Initiator Nonce", 	"ipv6.shim6.inonce",
				FT_UINT32, BASE_HEX, NULL, 0x0,
				"", HFILL }},

    { &hf_ipv6_shim6_rnonce,
      { "Responder Nonce", 	"ipv6.shim6.rnonce",
				FT_UINT32, BASE_HEX, NULL, 0x0,
				"", HFILL }},

    { &hf_ipv6_shim6_precvd,
      { "Probes Received", 	"ipv6.shim6.precvd",
				FT_UINT8, BASE_DEC, NULL, 0x0,
				"", HFILL }},

    { &hf_ipv6_shim6_psent,
      { "Probes Sent", 		"ipv6.shim6.psent",
				FT_UINT8, BASE_DEC, NULL, 0x0,
				"", HFILL }},

    { &hf_ipv6_shim6_reap,
      { "REAP State", 		"ipv6.shim6.reap",
				FT_UINT8, BASE_DEC, NULL, 0x0,
				"", HFILL }},

    { &hf_ipv6_shim6_opt_type,
      { "Option Type", 		"ipv6.shim6.opt.type",
				FT_UINT16, BASE_DEC, NULL, 0x0,
				"", HFILL }},

    { &hf_ipv6_shim6_opt_critical,
      { "Option Critical Bit", 	"ipv6.shim6.opt.critical",
				FT_BOOLEAN, BASE_NONE, NULL, 0x0,
				"", HFILL }},

    { &hf_ipv6_shim6_opt_len,
      { "Option Length",	"ipv6.shim6.opt.len",
				FT_UINT16, BASE_DEC, NULL, 0x0,
				"", HFILL }},

    { &hf_ipv6_shim6_opt_loclist,
      { "Locator List Generation", "ipv6.shim6.opt.loclist",
				FT_UINT32, BASE_HEX, NULL, 0x0,
				"", HFILL }},

    { &hf_ipv6_shim6_opt_locnum,
      { "Num Locators",		"ipv6.shim6.opt.locnum",
				FT_UINT8, BASE_DEC, NULL, 0x0,
				"Number of Locators in Locator List", HFILL }},

    { &hf_ipv6_shim6_opt_elemlen,
      { "Element Length",	"ipv6.shim6.opt.elemlen",
				FT_UINT8, BASE_DEC, NULL, 0x0,
				"Length of Elements in Locator Preferences Option", HFILL }},

    { &hf_ipv6_shim6_opt_fii,
      { "Forked Instance Identifier", "ipv6.shim6.opt.fii",
				FT_UINT32, BASE_HEX, NULL, 0x0,
				"", HFILL }},

#ifdef TEST_FINALHDR
    { &hf_ipv6_final,
      { "Final next header",	"ipv6.final",
				FT_UINT8, BASE_HEX, NULL, 0x0, "", HFILL }},
#endif
  };
  static gint *ett[] = {
    &ett_ipv6,
    &ett_ipv6_fragments,
    &ett_ipv6_fragment,
  };
  module_t *ipv6_module;

  proto_ipv6 = proto_register_protocol("Internet Protocol Version 6", "IPv6", "ipv6");
  proto_register_field_array(proto_ipv6, hf, array_length(hf));
  proto_register_subtree_array(ett, array_length(ett));

  /* Register configuration options */
  ipv6_module = prefs_register_protocol(proto_ipv6, NULL);
  prefs_register_bool_preference(ipv6_module, "defragment",
	"Reassemble fragmented IPv6 datagrams",
	"Whether fragmented IPv6 datagrams should be reassembled",
	&ipv6_reassemble);

  register_dissector("ipv6", dissect_ipv6, proto_ipv6);
  register_init_routine(ipv6_reassemble_init);
}

void
proto_reg_handoff_ipv6(void)
{
  dissector_handle_t ipv6_handle;

  data_handle = find_dissector("data");
  ipv6_handle = find_dissector("ipv6");
  dissector_add("ethertype", ETHERTYPE_IPv6, ipv6_handle);
  dissector_add("ppp.protocol", PPP_IPV6, ipv6_handle);
  dissector_add("ppp.protocol", ETHERTYPE_IPv6, ipv6_handle);
  dissector_add("gre.proto", ETHERTYPE_IPv6, ipv6_handle);
  dissector_add("ip.proto", IP_PROTO_IPV6, ipv6_handle);
  dissector_add("null.type", BSD_AF_INET6_BSD, ipv6_handle);
  dissector_add("null.type", BSD_AF_INET6_FREEBSD, ipv6_handle);
  dissector_add("null.type", BSD_AF_INET6_DARWIN, ipv6_handle);
  dissector_add("chdlctype", ETHERTYPE_IPv6, ipv6_handle);
  dissector_add("fr.ietf", NLPID_IP6, ipv6_handle);
  dissector_add("osinl.excl", NLPID_IP6, ipv6_handle);
  dissector_add("x.25.spi", NLPID_IP6, ipv6_handle);
  dissector_add("arcnet.protocol_id", ARCNET_PROTO_IPv6, ipv6_handle);

  ip_dissector_table = find_dissector_table("ip.proto");
}
