/*
 * (c) 2007 by Matthijs Mekking / NLnet Labs
 *
 * SHIM6 Library
 *
 */

#include "thc-ipv6.h"
#include "sim-shim6.h"

#include <byteswap.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>

int sim_debug = 1;

unsigned int compute_checksum(unsigned char *hdr, int len, unsigned int csum) {
  unsigned int csum16;
  int i;  

  for (i=0; i<len; i += 2) {
    csum += (hdr[i]*256 + hdr[i+1]);
  }
  csum16 = csum/(256*256) + csum%(256*256);
  csum16 = ~csum16;

  return csum16;
}

/* Add Payload */

int sim_add_dummytcp(unsigned char *pkt, int *pkt_len, unsigned char *data, int len) {
  thc_ipv6_hdr *hdr = (thc_ipv6_hdr *) pkt;
  thc_ipv6_ext_hdr *ehdr = (thc_ipv6_ext_hdr *) hdr->final;
  unsigned char *buf = malloc(len);

  if (hdr == NULL || buf == NULL)
    return -1;

  return 0;
}

int sim_add_payload(unsigned char *pkt, int *pkt_len, unsigned char *ct) {
  thc_ipv6_hdr *hdr = (thc_ipv6_hdr *) pkt;
  thc_ipv6_ext_hdr *ehdr = (thc_ipv6_ext_hdr *) hdr->final, *nehdr = malloc(sizeof(thc_ipv6_ext_hdr));
  unsigned char *buf = malloc(6);

  if (nehdr == NULL || hdr == NULL || buf == NULL)
    return -1;

  if (ehdr == NULL) {
    hdr->next = NXT_SHIM6;
    hdr->next_segment = (char *) nehdr;
  } else {
    ehdr->next = NXT_SHIM6;
    ehdr->next_segment = (char *) nehdr;
  }
  hdr->final = (char *) nehdr;
  hdr->final_type = NXT_SHIM6;

  memset(buf, 0, 6);
  memcpy(buf, ct, 6);
  
  /* set the P field to 1 */
  buf[0] = buf[0] + 128;

  nehdr->next_segment = NULL;
  nehdr->next = NXT_NONXT;
  nehdr->length = 0; 
  nehdr->data = buf;
  nehdr->data_len = 6;
  hdr->length += nehdr->data_len + 2;
  *pkt_len += nehdr->data_len + 2;
  
  if (sim_debug)
    fprintf(stderr, "SHIM6 Payload header added succesfully\n"); 

  return 0;
}

/* Add Options */
int sim_add_opts(unsigned char *pkt, int *pkt_len, unsigned char type, int c, int len, unsigned char *contents) {
  thc_ipv6_hdr *hdr = (thc_ipv6_hdr *) pkt;
  thc_ipv6_ext_hdr *ehdr = (thc_ipv6_ext_hdr *) hdr->final;
  unsigned char *buf;
  int offset, total_len;
  unsigned int csum, csum16;

  if (ehdr == NULL || hdr == NULL)
    return -1;
  if (hdr->final_type != NXT_SHIM6) {
    fprintf(stderr, "Invalid packet to add SHIM6 options\n"); 
    return -1;
  }

  hdr->final = (char *) ehdr;

  total_len = (11+len) - ((len+3) % 8);
  buf = malloc(total_len+ehdr->data_len);
  offset = ehdr->data_len;
  memcpy(buf, ehdr->data, ehdr->data_len);

  /* Type */
  buf[offset] = 0;
  buf[offset+1] = (type << 1);
  if (c == CRITICAL)
    buf[offset+1]++;
  
  /* Content Length */  
  buf[offset+2] = len / 256;
  buf[offset+3] = len % 256;

  if (type == SHIM6_OPT_LOCLIST) {
    int padd = len - 5; 
    padd = padd % 17;
    printf("padding is %u\n", padd); 

    buf[offset+2] = (len-padd) / 256;
    buf[offset+3] = (len-padd) % 256;
  }

  /* Adding content */
  memcpy(&buf[offset+4], contents, len);

  /* Padding */
  while (len+4 < total_len) {
    buf[offset+4+len] = 0;
    len++;
  }
  
  buf[2] = 0;
  buf[3] = 0;

  csum = ehdr->next*256 + ehdr->length+(total_len >> 3);
  csum16 = compute_checksum(buf, ehdr->data_len + total_len, csum);

  buf[2] = csum16 >> 8;
  buf[3] = csum16%256;

  ehdr->data = buf;
  ehdr->length += (total_len >> 3);
  ehdr->data_len += total_len;

  hdr->length += total_len;
  *pkt_len += total_len;

  if (sim_debug)
    fprintf(stderr, "SHIM6 Option (%u) added succesfully\n", type); 

  return 0;
}

int sim_add_opts_respval(unsigned char *pkt, int *pkt_len, int c, unsigned char *val, int len) {

  return sim_add_opts(pkt, pkt_len, SHIM6_OPT_RESPVAL, c, len, val); 
}

int sim_add_opts_loclist(unsigned char *pkt, int *pkt_len, int c, unsigned char *llg, struct shim_loc_list *list) {
  struct shim_loc_list *nxt;
  unsigned char *buf;
  int i, padd, offset, count;

  i = 0;
  nxt = list;
  while (nxt != NULL) {
	i++;
	nxt = nxt->next;
  }

  padd = (7 - i) % 8;
  buf = malloc(i*17 + padd + 5);
  memset(buf, 0, i*17+padd+5);

  memcpy(&buf[0], llg, 4);		/* locator list generation */
  memset(&buf[4], i, 1);		/* num locators */

  /* padding */
  offset = 5;
  if (padd>0)
   	memset(&buf[offset+i], 0, padd);

  /* locator lists, verification methods */
  count = 0;
  nxt = list;

  while (nxt != NULL) {
	memset(&buf[offset+count], nxt->verif_method, 1);
	memcpy(&buf[offset+padd+i+count*16], nxt->locator, 16);
	nxt = nxt->next;
	count++;
  }

  return sim_add_opts(pkt, pkt_len, SHIM6_OPT_LOCLIST, c, i*17+padd+5, buf); 
}

int sim_add_opts_locpref(unsigned char *pkt, int *pkt_len, int c, unsigned char *llg, int ellen, struct shim_loc_list *list) {
  struct shim_loc_list *nxt;
  unsigned char *buf;
  int i, padd, offset, count;

  i = 0;
  nxt = list;
  while (nxt != NULL) {
	i++;
	nxt = nxt->next;
  }

  buf = malloc(ellen*i + 5);
  memset(buf, 0, ellen*i + 5);

  memcpy(&buf[0], llg, 4);		/* locator list generation */
  memset(&buf[4], ellen, 1);		/* element len */

  if (ellen > 3)
	ellen = 3;

  /* locator lists, verification methods */
  offset = 5; 
  count = 0;
  nxt = list;

  while (nxt != NULL) {
	if (ellen > 0) 
	    memset(&buf[offset+count++], nxt->flag, 1);
	if (ellen > 1) 
	    memset(&buf[offset+count++], nxt->priority, 1);
	if (ellen > 2) 
	    memset(&buf[offset+count++], nxt->weight, 1);

	nxt = nxt->next;
  }

  return sim_add_opts(pkt, pkt_len, SHIM6_OPT_LOCPREF, c, ellen*i+5, buf); 
}

int sim_add_opts_cgapdm(unsigned char *pkt, int *pkt_len, int c, unsigned char *pdm, int len) {

  return sim_add_opts(pkt, pkt_len, SHIM6_OPT_CGAPDM, c, len, pdm); 
}

int sim_add_opts_cgasig(unsigned char *pkt, int *pkt_len, int c, unsigned char *sig, int len) {

  return sim_add_opts(pkt, pkt_len, SHIM6_OPT_CGASIG, c, len, sig); 
}

// cga signature

int sim_add_opts_ulidpair(unsigned char *pkt, int *pkt_len, int c, unsigned char *snd, unsigned char *rcv) {
  unsigned char buf[36];
  memset(buf, 0, 36); 		/* 4 bytes reserved */
  memcpy(&buf[4], snd, 16);	/* 16 bytes ulid sender */
  memcpy(&buf[20], rcv, 16);	/* 16 bytes ulid receiver */

  return sim_add_opts(pkt, pkt_len, SHIM6_OPT_ULIDPAIR, c, 36, buf); 
}

int sim_add_opts_fii(unsigned char *pkt, int *pkt_len, int c, unsigned char *fii) {
  return sim_add_opts(pkt, pkt_len, SHIM6_OPT_FII, c, 4, fii); 
}

int sim_add_opts_illegal(unsigned char *pkt, int *pkt_len, int c, unsigned char *illegal, int len) {
  return sim_add_opts(pkt, pkt_len, SHIM6_OPT_ILLEGAL, c, len, illegal); 
}

/* Add Control Messages */

int sim_add_control(unsigned char *pkt, int *pkt_len, unsigned char type, unsigned char *buf2, int len) {
  int i;
  unsigned int csum;
  int csum16;
  thc_ipv6_hdr *hdr = (thc_ipv6_hdr *) pkt;
  thc_ipv6_ext_hdr *ehdr = (thc_ipv6_ext_hdr *) hdr->final, *nehdr = malloc(sizeof(thc_ipv6_ext_hdr));

  unsigned char *buf = malloc(len+4);

  if (nehdr == NULL || hdr == NULL)
    return -1;

  if (ehdr == NULL) {
    hdr->next = NXT_SHIM6;
    hdr->next_segment = (char *) nehdr;
  } else {
    ehdr->next = NXT_SHIM6;
    ehdr->next_segment = (char *) nehdr;
  }
  hdr->final = (char *) nehdr;
  hdr->final_type = NXT_SHIM6;

  memset(buf, 0, len+4);
  buf[0] = type; /* P = 0, type */
  buf[1] = 0;	 /* reserved + zero bit */

  /* Adding elements (initiator context tag, initiator nonce) */
  memcpy(&buf[4], buf2, len);

  /* computing checksum */
  csum = NXT_NONXT*256 + (((len+2+4) - 8) >> 3);

  /* bad checksum */
  /* csum = NXT_NONXT*254 + (((len+2+4) - 8) >> 3); */

  csum16 = compute_checksum((unsigned char *) buf, len+4, csum);
  
  buf[2] = csum16 >> 8;
  buf[3] = csum16%256;

  nehdr->next_segment = NULL;
  nehdr->next = NXT_NONXT;
  nehdr->length = ((len+2+4) - 8) >> 3;

  /* try buffer overflow */
  /* nehdr->length = ((len+2+400) - 8) >> 3; */

  nehdr->data = buf;
  nehdr->data_len = len+4;
  hdr->length += nehdr->data_len + 2;
  *pkt_len += nehdr->data_len + 2;
  
  if (sim_debug)
    fprintf(stderr, "SHIM6 Control message added succesfully\n"); 

  return 0;
}

int sim_add_control_i1(unsigned char *pkt, int *pkt_len, unsigned char *ct, unsigned char *nonce, int len) {
  unsigned char *buf = malloc(len);
  
  /* Adding elements (initiator context tag, initiator nonce) */
  memset(buf, 0, len); 
  memcpy(&buf[0], ct, 6);
  memcpy(&buf[6], nonce, 4);
  
  buf[0] = buf[0] & SHIM6_BITMASK_CT; 

  return sim_add_control(pkt, pkt_len, SHIM6_TYPE_I1, buf, len); 
}

int sim_add_control_r1(unsigned char *pkt, int *pkt_len, unsigned char *nonce, unsigned char *nonce2, int len) {
  unsigned char *buf = malloc(len);
  
  /* Adding elements (reserved2, initiator nonce, responder nonce) */
  memset(buf, 0, len);
  memcpy(&buf[2], nonce, 4);
  memcpy(&buf[6], nonce2, 4);

  return sim_add_control(pkt, pkt_len, SHIM6_TYPE_R1, buf, len); 
}

int sim_add_control_i2(unsigned char *pkt, int *pkt_len, unsigned char *ct, unsigned char *nonce, unsigned char *nonce2, int len) {
  unsigned char *buf = malloc(len);
  
  /* Adding elements (initiator context tag, initiator nonce, responder nonce, reserved2) */
  memset(buf, 0, len); 
  memcpy(&buf[0], ct, 6);
  memcpy(&buf[6], nonce, 4);
  memcpy(&buf[10], nonce2, 4);

  buf[0] = buf[0] & SHIM6_BITMASK_CT;

  return sim_add_control(pkt, pkt_len, SHIM6_TYPE_I2, buf, len); 
}

int sim_add_control_r2(unsigned char *pkt, int *pkt_len, unsigned char *ct, unsigned char *nonce, int len) {
  unsigned char *buf = malloc(len);
  
  /* Adding elements (responder context tag, initiator nonce) */
  memset(buf, 0, len); 
  memcpy(&buf[0], ct, 6);
  memcpy(&buf[6], nonce, 4);
  
  buf[0] = buf[0] & SHIM6_BITMASK_CT; 

  return sim_add_control(pkt, pkt_len, SHIM6_TYPE_R2, buf, len); 
}

int sim_add_control_r1bis(unsigned char *pkt, int *pkt_len, unsigned char *ct, unsigned char *nonce, int len) {
  unsigned char *buf = malloc(len);
  
  /* Adding elements (responder nonce, initiator nonce) */
  memcpy(&buf[0], ct, 6);
  memcpy(&buf[6], nonce, 4);

  buf[0] = buf[0] & SHIM6_BITMASK_CT;

  return sim_add_control(pkt, pkt_len, SHIM6_TYPE_R1BIS, buf, len); 
}

int sim_add_control_i2bis(unsigned char *pkt, int *pkt_len, unsigned char *ct, unsigned char *ct2, unsigned char *nonce, unsigned char *nonce2, int len) {
  unsigned char *buf = malloc(len);
  
  /* Adding elements (initiator context tag, initiator nonce, responder nonce, reserved2) */
  memset(buf, 0, len); 
  memcpy(&buf[0], ct, 6);
  memcpy(&buf[6], nonce, 4);
  memcpy(&buf[10], nonce2, 4);
  /* reserved 6 bytes + 1 bit */
  memcpy(&buf[20], ct2, 6);

  buf[0] = buf[0] & SHIM6_BITMASK_CT;

  return sim_add_control(pkt, pkt_len, SHIM6_TYPE_I2BIS, buf, len); 
}

int sim_add_control_updreq(unsigned char *pkt, int *pkt_len, unsigned char *ct, unsigned char *nonce, int len) {
  unsigned char *buf = malloc(len);
  
  printf("adding ct\n"); 

  /* Adding elements (request context tag, request nonce) */
  memcpy(&buf[0], ct, 6);

  printf("adding nonce\n"); 
  memcpy(&buf[6], nonce, 4);

  printf("adding bitmask\n"); 

  buf[0] = buf[0] & SHIM6_BITMASK_CT;

  printf("adding control message\n"); 

  return sim_add_control(pkt, pkt_len, SHIM6_TYPE_UPDREQ, buf, len); 
}

int sim_add_control_updack(unsigned char *pkt, int *pkt_len, unsigned char *ct, unsigned char *nonce, int len) {
  unsigned char *buf = malloc(len);
  
  /* Adding elements (receiver context tag, request nonce) */
  memcpy(&buf[0], ct, 6);
  memcpy(&buf[6], nonce, 4);

  buf[0] = buf[0] & SHIM6_BITMASK_CT;

  return sim_add_control(pkt, pkt_len, SHIM6_TYPE_UPDACK, buf, len); 
}

int sim_add_control_keepalive(unsigned char *pkt, int *pkt_len, unsigned char *ct) {
  unsigned char *buf = malloc(10);
  
  /* Adding elements (receiver context tag, reserved) */
  memset(buf, 0, 10); 
  memcpy(&buf[0], ct, 6);

  buf[0] = buf[0] & SHIM6_BITMASK_CT;

  return sim_add_control(pkt, pkt_len, SHIM6_TYPE_KEEPALIVE, buf, 10); 
}

int sim_add_control_probe(unsigned char *pkt, int *pkt_len, unsigned char *ct, int sta, struct shim_probe_list *recvd, struct shim_probe_list *sent) {
  struct shim_probe_list *nxt;
  unsigned char *buf;
  int psent, precvd, offset, count;

  psent = 0;
  nxt = sent;
  while (nxt != NULL) {
	psent++;
	nxt = nxt->next;
  }

  precvd = 0;
  nxt = recvd;
  while (nxt != NULL) {
	precvd++;
	nxt = nxt->next;
  }

  buf = malloc(40*(psent+precvd)+10);
  memset(buf, 0, 40*(psent+precvd)+10);

  /* Adding elements (receiver context tag) */
  memcpy(&buf[0], ct, 6);
  buf[6] = psent*16 + precvd;	/* psent & precvd */
  buf[7] = sta >> 6;		/* sta, REAP State */
  /* byte 7, 8, 9 are Reserved2 */

  /* Probes Sent */
  offset = 10; 
  count = 0;
  nxt = sent;

  while (nxt != NULL) {
	memcpy(&buf[offset], nxt->source, 16); 
	memcpy(&buf[offset+16], nxt->destination, 16); 
	memcpy(&buf[offset+32], nxt->nonce, 4); 
	memcpy(&buf[offset+36], nxt->data, 4); 
	nxt = nxt->next;

	offset += 40;
  }

  /* Probes Received */
  count = 0;
  nxt = recvd;

  while (nxt != NULL) {
	memcpy(&buf[offset], nxt->source, 16); 
	memcpy(&buf[offset+16], nxt->destination, 16); 
	memcpy(&buf[offset+32], nxt->nonce, 4); 
	memcpy(&buf[offset+36], nxt->data, 4); 
	nxt = nxt->next;

	offset += 40;
  }

  buf[0] = buf[0] & SHIM6_BITMASK_CT;

  if (sim_debug)
	thc_dump_data(buf, 40*(psent+precvd)+10, "Probe packet: "); 

  return sim_add_control(pkt, pkt_len, SHIM6_TYPE_PROBE, buf, 40*(psent+precvd)+10); 
}

const char *val_to_str(int value, const value_string *vs) {
  const char *result;
  int i = 0;

  while (vs[i].strptr) {
    if (vs[i].value == value) {
	return vs[i].strptr; 
    }
    i++;
  }

  return vs[0].strptr;
}

static const value_string nxt_hdr_vals[] = {
  {0, "Some Payload"}, /* default is payload header */
  {NXT_NONXT, "No Next Header"}, 
  {NXT_SHIM6, "Shim6 Header"}, 
  {-1, NULL}
};

static const value_string shim_ctrl_vals[] = {
  {0, "Unkown Control Message Type"},
  {SHIM6_TYPE_I1, "I1"}, 
  {SHIM6_TYPE_R1, "R1"}, 
  {SHIM6_TYPE_I2, "I2"}, 
  {SHIM6_TYPE_R2, "R2"}, 
  {SHIM6_TYPE_R1BIS, "R1bis"}, 
  {SHIM6_TYPE_I2BIS, "I2bis"}, 
  {SHIM6_TYPE_UPDREQ, "Update Request"}, 
  {SHIM6_TYPE_UPDACK, "Update Acknowledgement"}, 
  {SHIM6_TYPE_PROBE, "Probe"}, 
  {SHIM6_TYPE_KEEPALIVE, "Keepalive"}, 
  {-1, NULL}
};


int sim_print_shim6(unsigned char *pkt) {
  int i; 
  char *tmp; 

  if (pkt == NULL)
	return 0;
  
  /* shim6 part starts at 6+34 */
  if ((pkt[42] & SHIM6_BITMASK_P) == 128) 
    printf("Payload Extension Header\n"); 
  else {
    printf("Control Message"); 
    tmp = pkt[42] % ~(SHIM6_BITMASK_P); 
    printf("(%s)\n", val_to_str(tmp, shim_ctrl_vals)); 
  }

  printf("\n"); 

  return 1;
}
