/*
 *  shimterceptor: A Shim6 Interceptor at user level
 *  Matthijs Mekking (c) 2007
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/wait.h>
#include <time.h>
#include <pcap.h>
#include "thc-ipv6.h"
#include "sim-shim6.h"
#include "sim-ctx.h"

/* debugging */

extern int debug;
extern int sim_debug;
int opt_critical = 0;
int r1_received = 0;

/* global variables */
unsigned char *pkt = NULL;
int pkt_len = 0;
thc_ipv6_hdr *ipv6;
char *interface;
char *multicast = "ff020000000000000000000000000001";
char *spoof = "2001000000000000020c29fffeb97fb6";
char *my_ipv6;

/* shim6 variables */
struct shim_ctx *ctx; 
struct shim_loc_list *top = NULL; 
struct shim_loc_list *stack = NULL; 
int i1sent = 0;
int i2sent = 0;
int i2bissent = 0;

/* HELP FUNCTIONS */

void add_locator(unsigned char *ulid_local) {
  if (stack == NULL) {
	struct shim_loc_list *new_locator; 
	new_locator = (struct shim_loc_list *) malloc(sizeof(struct shim_loc_list));
	new_locator->locator = ulid_local;
	new_locator->verif_method = 0; 
	new_locator->flag = SHIM6_PREF_BROKEN; 
	new_locator->priority = 1; 
	new_locator->weight = 2; 
  	new_locator->next = NULL;
		  
	stack = new_locator;
	top = stack;
  }
  else {
 	struct shim_loc_list *new_locator; 
	new_locator = (struct shim_loc_list *) malloc(sizeof(struct shim_loc_list));
	new_locator->locator = ulid_local;
	new_locator->verif_method = 0; 
	new_locator->flag = 0;
	new_locator->priority = 2;
	new_locator->weight = 1;
  	new_locator->next = NULL;

	top->next = new_locator;
	top = new_locator;
  }
  return;
}

int is_local_addr(unsigned char *ipv6hdr, int start) {
   struct shim_loc_list *new_locator = (struct shim_loc_list *) malloc(sizeof(struct shim_loc_list));
   new_locator = stack;

   while (new_locator != NULL) {
	if (memcmp(&ipv6hdr[start], new_locator->locator, 16) == 0)
		return 1;
	new_locator = new_locator->next; 
   }		

   return 0;
}

int is_peer_addr(unsigned char *ipv6hdr, int start) {
   struct shim_loc_list *new_locator = (struct shim_loc_list *) malloc(sizeof(struct shim_loc_list));
	
   if (ctx != NULL) {
	   new_locator = ctx->cpeerls;

	   while (new_locator != NULL) {
		if (memcmp(&ipv6hdr[start], new_locator->locator, 16) == 0)
			return 1;
		new_locator = new_locator->next; 
	   }		
   }


   return 0;
}

/* LOOKUP FUNCTIONS */

int lookup_ctx_ulidpair(unsigned char *ipv6hdr, int start, int options, unsigned char *sender, unsigned char *receiver) {
	int i, len;
	int option = 0;

	if (ipv6hdr[start+1] > 1) {
		/* dissect options */
		i = options;

                while (i<(ipv6hdr[start+1]*8 + 8)) {
			option = ipv6hdr[start+i]*256 + (ipv6hdr[start+i+1]>>1);
			if (option == SHIM6_OPT_ULIDPAIR) {
			  if (memcmp(sender, &ipv6hdr[start+i+8], 16) == 0)
			    if (memcmp(receiver, &ipv6hdr[start+i+24], 16) == 0) 
				return 1;
                        }
			len = ipv6hdr[start+i+2]*256 + ipv6hdr[start+i+3];
                        i += 11 + len - ((len+3)%8);
                }
        }

        if (option != SHIM6_OPT_ULIDPAIR) {
	  if (memcmp(sender, &ipv6hdr[8], 16) == 0)
	    if (memcmp(receiver, &ipv6hdr[24], 16) == 0)
		return 1;
        }
	
	return 0;
}

int lookup_ctx_fii(unsigned char *fii) {
	if (ctx == NULL)
		return 0;
	if (ctx->cfii == NULL && fii == NULL)
		return 1;
	if (ctx->cfii == NULL || fii == NULL)
		return 0;

	/* non-NULL values, you may memcmp */
	return (memcmp(ctx->cfii, fii, 4) == 0);
}


/* actually this function must check if lp fall in the locator sets. This is impossible for the peer 
   locator, since in i1-sent, the peer locator set is not stored. So this function now checks if the 
   lp is equal to the ulid pair or the lp pair, the only information of the responder stored. */
int lp_in_ls(unsigned char *sender, unsigned char *receiver) {
	if (ctx == NULL)
		return 0;
	if (receiver == NULL || sender == NULL || ctx->clocallp == NULL || ctx->cpeerlp == NULL)
		return 0;

	if (memcmp(ctx->clocallp, receiver, 16) == 0 && memcmp(ctx->cpeerlp, sender, 16) == 0);
		return 1;

	return (memcmp(ctx->clocalulid, receiver, 16) == 0 && memcmp(ctx->cpeerulid, sender, 16) == 0);
}

int lookup_ctx_lp(unsigned char *sender, unsigned char *receiver) {
	if (ctx == NULL)
		return 0;

	return (memcmp(ctx->clocallp, receiver, 16) == 0 && memcmp(ctx->cpeerlp, sender, 16) == 0);
}

int lookup_ctx_peertag(unsigned char *ipv6hdr, int start) {
	unsigned char *tag; 

        if (ctx == NULL) 
		return 0;

	memcpy(tag, &ipv6hdr[start], 6); 
	tag[0] = tag[0] % SHIM6_BITMASK_CT;

	return (memcmp(ctx->cpeertag, tag, 6) == 0); 
}

int lookup_ctx_nonceinit(unsigned char *ipv6hdr, int start) {
	unsigned char *non; 

        if (ctx == NULL) 
		return 0;

	memcpy(non, &ipv6hdr[start], 4); 

	return (memcmp(ctx->cnonceinit, non, 4) == 0); 
}

/* STORE FUNCTIONS */

void store_ctx_state(int sta) {
	ctx->cstate = sta; 
}

void store_ctx_ulidpair(unsigned char *ipv6hdr, int start, int options) {
	unsigned char *addr; 
	int i, len;
	int option = 0;

	if (ipv6hdr[start+1] > 1) {
		/* dissect options */
		i = options;

                while (i<(ipv6hdr[start+1]*8 + 8)) {
			option = ipv6hdr[start+i]*256 + (ipv6hdr[start+i+1]>>1);
			if (option == SHIM6_OPT_ULIDPAIR) {
				addr = malloc(16);
                                memcpy(addr, &ipv6hdr[start+i+8], 16);
                                ctx->clocalulid = addr;

                                addr = malloc(16);
                                memcpy(addr, &ipv6hdr[start+i+24], 16);
                                ctx->cpeerulid = addr;
                                break;
                        }
			len = ipv6hdr[start+i+2]*256 + ipv6hdr[start+i+3];
                        i += 11 + len - ((len+3)%8);
                }
        }

        if (option != SHIM6_OPT_ULIDPAIR) {
                addr = malloc(16);
                memcpy(addr, &ipv6hdr[8], 16);
                ctx->clocalulid = addr;

                addr = malloc(16);
                memcpy(addr, &ipv6hdr[24], 16);
                ctx->cpeerulid = addr;
        }
}

void store_ctx_fii(unsigned char *ipv6hdr, int start, int options) {
	unsigned char *addr; 
	int i, len;
	int option = 0;

	if (ipv6hdr[start+1] > 1) {
		/* dissect options */
		i = options;

                while (i<(ipv6hdr[start+1]*8 + 8)) {
			option = ipv6hdr[start+i]*256 + (ipv6hdr[start+i+1]>>1);
			if (option == SHIM6_OPT_FII) {
				addr = malloc(4);
                                memcpy(addr, &ipv6hdr[start+i+4], 4);
                                ctx->cfii = addr;
                                break;
                        }
			len = ipv6hdr[start+i+2]*256 + ipv6hdr[start+i+3];
                        i += 11 + len - ((len+3)%8);
                }
        }
}

void store_ctx_lp(unsigned char *ipv6hdr, int local, int peer) {
	unsigned char *addr;

	addr = malloc(16);
	memcpy(addr, &ipv6hdr[local], 16); 
	ctx->clocallp = addr; 

	addr = malloc(16);
	memcpy(addr, &ipv6hdr[peer], 16); 
	ctx->cpeerlp = addr; 
}

void store_ctx_lslocal(struct shim_loc_list *list) {
	int j;

	unsigned char *listgen = malloc(4); 

	for (j=0;j<4;j++)
		listgen[j] = rand();

	ctx->clocalls = list;
	ctx->clocallistgen = listgen;
}

void store_ctx_lspeer(unsigned char *ipv6hdr, int type, int start, int options) {
	int i, j, len, nr, offset;
	int option = 0;
	struct shim_loc_list *ptop = NULL; 
	unsigned char *tmp;

	if ((type == SHIM6_TYPE_R2 && ipv6hdr[start+1] > 1) || 
	    (type == SHIM6_TYPE_I2 && ipv6hdr[start+1] > 2) || 
	    (type == SHIM6_TYPE_I2BIS && ipv6hdr[start+1] > 3)) {
		/* dissect options */
		i = options;

                while (i<(ipv6hdr[start+1]*8 + 8)) {
			option = ipv6hdr[start+i]*256 + (ipv6hdr[start+i+1]>>1);
			len = ipv6hdr[start+i+2]*256 + ipv6hdr[start+i+3];

			if (option == SHIM6_OPT_LOCLIST) {
				tmp = malloc(4); 	
				memcpy(tmp, &ipv6hdr[start+i+4], 4);
				ctx->clocallistgen = tmp;
				nr = ipv6hdr[start+i+8];
				j = 1;
				ctx->cpeerls = NULL;

				while ((j<=nr) && (i+17*nr+((7-nr)%8) <(ipv6hdr[start+1]*8 + 8))) {
				  struct shim_loc_list *new_locator; 
				  new_locator = (struct shim_loc_list *) malloc(sizeof(struct shim_loc_list));

				  offset = start+i+8+nr+((7-nr)%8)+(16*(j-1));
				  tmp = malloc(16);
				  memcpy(tmp, &ipv6hdr[offset], 16);
				  new_locator->locator = tmp;

				  offset = start+i+8+j;
				  new_locator->verif_method = ipv6hdr[offset]; 

 				  if (ctx->cpeerls == NULL) {
					new_locator->next = NULL;
		  			ctx->cpeerls = new_locator;
					ptop = ctx->cpeerls;
				  }
				  else {
					ptop->next = new_locator;
					ptop = new_locator;
				  }
				j++;
				}

				break;
                        }
                        i += 11 + len - ((len+3)%8);
                }
        }
}

void store_ctx_taglocal(unsigned char *ipv6hdr, int start) {
	unsigned char *ctag = malloc(6);
	memcpy(ctag, &ipv6hdr[start], 6); 

	ctag[0] = ctag[0] % SHIM6_BITMASK_CT;
	ctx->clocaltag = ctag;
}

void store_ctx_tagpeer(unsigned char *ipv6hdr, int start) {
	unsigned char *ctag = malloc(6);
	memcpy(ctag, &ipv6hdr[start], 6); 
	printf("ctag including p tag: %s\n", thc_2string(ctag, 6)); 

	ctag[0] = ctag[0] % SHIM6_BITMASK_CT;
	printf("ctag excluding p tag: %s\n", thc_2string(ctag, 6)); 

	ctx->cpeertag = ctag;
}

void store_ctx_nonceinit(unsigned char *ipv6hdr, int start) {
	unsigned char *non = malloc(4);
	memcpy(non, &ipv6hdr[start], 4);
	ctx->cnonceinit = non;
}

void store_ctx_nonceresp(unsigned char *ipv6hdr, int start) {
	unsigned char *non = malloc(4);
	memcpy(non, &ipv6hdr[start], 4);
	ctx->cnonceresp = non;
}

void store_ctx_respval(unsigned char *ipv6hdr, int type, int start, int options) {
	unsigned char *rval; 
	int i, len;
	int option = 0;

	if ((type == SHIM6_TYPE_R1 && ipv6hdr[start+1] > 1) || 
	    (type == SHIM6_TYPE_R1BIS && ipv6hdr[start+1] > 1) || 
	    (type == SHIM6_TYPE_I2 && ipv6hdr[start+1] > 2) || 
	    (type == SHIM6_TYPE_I2BIS && ipv6hdr[start+1] > 3)) {
		/* dissect options */
		i = options;

                while (i<(ipv6hdr[start+1]*8 + 8)) {
			option = ipv6hdr[start+i]*256 + (ipv6hdr[start+i+1]>>1);
			len = ipv6hdr[start+i+2]*256 + ipv6hdr[start+i+3];

			if (option == SHIM6_OPT_RESPVAL) {
				rval = malloc(len);
                                memcpy(rval, &ipv6hdr[start+i+4], len);
                                ctx->crespval = rval;
                                ctx->crespvallen = len;
                                break;
                        }
                        i += 11 + len - ((len+3)%8);
                }
        }
}

void store_reap_state(int sta) {
	ctx->rstate = sta; 
}

/* PRINT FUNCTIONS */

int help(char *prg) {
  printf("USAGE\n");
  printf("      %s <interface>\n", prg);
  printf("DESCRIPTION\n");
  printf("     captures shim6 messages on <interface>\n");
  return 1;
}

static const value_string shim_ctx_states[] = {
  {0, "Unknown Context State"}, 
  {SHIM6_STATE_IDLE, "Idle"}, 
  {SHIM6_STATE_I1SENT, "I1-Sent"}, 
  {SHIM6_STATE_I2SENT, "I2-Sent"}, 
  {SHIM6_STATE_I2BISSENT, "I2bis-Sent"}, 
  {SHIM6_STATE_ESTABLISHED, "Established"}, 
  {SHIM6_STATE_NOSUPPORT, "No-Support"}, 
  {SHIM6_STATE_EFAILED, "E-Failed"}, 
  {-1, NULL}
};

int print_ctx() {
  /*

   struct shim_loc_list *new_locator; 

   if (ctx == NULL) {
	printf("No context stored\n\n"); 
	return 0;
   }

   printf("Context\n");
   printf(" State:\n  %s\n", val_to_str(ctx->cstate, shim_ctx_states));
   if (ctx->cfii)
   printf(" Forked Instance Identifier:\n  %u\n", ctx->cfii);
   if (ctx->clocaltag)
   printf(" Context Tag local:\n  %s\n", thc_2string(ctx->clocaltag, 6));
   if (ctx->cpeertag)
   printf(" Context Tag peer:\n  %s\n", thc_2string(ctx->cpeertag, 6));
   printf(" ULID local:\n  %s\n", (unsigned char *) thc_string2notation(thc_ipv62string(ctx->clocalulid)));
   printf(" ULID peer:\n  %s\n", (unsigned char *) thc_string2notation(thc_ipv62string(ctx->cpeerulid)));
   printf(" Lp local:\n  %s\n", (unsigned char *) thc_string2notation(thc_ipv62string(ctx->clocallp)));
   printf(" Lp peer:\n  %s\n", (unsigned char *) thc_string2notation(thc_ipv62string(ctx->cpeerlp)));
   printf(" Ls local:\n"); 

   new_locator = (struct shim_loc_list *) malloc(sizeof(struct shim_loc_list));
   new_locator = ctx->clocalls; 

   while (new_locator != NULL) {
 	printf("  %s\n", (unsigned char *) thc_string2notation(thc_ipv62string(new_locator->locator)));
	new_locator = new_locator->next; 
   }		

   printf(" Ls peer:\n"); 

   new_locator = (struct shim_loc_list *) malloc(sizeof(struct shim_loc_list));
   new_locator = ctx->cpeerls; 

   while (new_locator != NULL) {
 	printf("  %s\n", (unsigned char *) thc_string2notation(thc_ipv62string(new_locator->locator)));
	new_locator = new_locator->next; 
   }		

   if (ctx->cnonceinit)
   printf(" Init Nonce:\n  %s\n", thc_2string(ctx->cnonceinit, 4));
   if (ctx->cnonceresp)
   printf(" Resp Nonce:\n  %s\n", thc_2string(ctx->cnonceresp, 4));
   if (ctx->crespval)
   printf(" Resp Validator:\n  %s\n", thc_2string(ctx->crespval, ctx->crespvallen));

   printf("REAP:\n"); 

   printf("\n"); 
  */
   return 1;
}

/* CAPTURE PACKETS */

void intercept(u_char *foo, const struct pcap_pkthdr *header, const unsigned char *data) {
  unsigned char *ipv6hdr = (unsigned char *)(data + 14);  
  unsigned int option;  
  unsigned char *addr; 
  unsigned char *src; 
  unsigned char *dst; 

  unsigned char *ctag = malloc(6);
  unsigned char *nonce1 = malloc(4);
  unsigned char *nonce2 = malloc(4);
  unsigned char *respval = malloc(10);
  unsigned char *ulidl = malloc(16);
  unsigned char *ulidp = malloc(16);
  unsigned char *listgen = malloc(4);

  int i, j, start;

  if (debug) {
    printf("DEBUG: packet received\n");
    thc_dump_data((unsigned char *)data, header->caplen, "Received Packet");
  }

  i = 6; /* IPv6 next header */
  if (ipv6hdr[i] != NXT_SHIM6) {
    return;
  }
  
  if (sim_debug) {
    thc_dump_data((unsigned char *)data, header->caplen, "Received Packet");
  }

  if (is_local_addr(ipv6hdr, 8)) 
  	printf("Sent Shim6 Packet: "); 
  else
  	printf("Received Shim6 Packet: "); 
  sim_print_shim6(ipv6hdr); 

  start = 40; /* Start shim6 extension header */

  /* generate ipv6 packet */
  src = malloc(16); 
  dst = malloc(16); 
  if (ctx == NULL) {
	/* src and dst from message */
	  memcpy(src, &ipv6hdr[24], 16); 
	  memcpy(dst, &ipv6hdr[8], 16); 
  }
  else {
	if (ctx->clocallp == NULL)
	  memcpy(src, my_ipv6, 16); 
	else
	  memcpy(src, ctx->clocallp, 16); 

	if (ctx->cpeerlp == NULL)
	  memcpy(dst, &ipv6hdr[8], 16); 
	else
	  memcpy(dst, ctx->cpeerlp, 16); 
  }

  if ((pkt = (unsigned char *) thc_create_ipv6(interface, PREFER_GLOBAL, &pkt_len,
		(unsigned char *) src, (unsigned char *) dst,
		255, 0, 0, 0, 0)) == NULL) {
	fprintf(stdout, "Cannot create IPv6 packet. Possible illegal interface or destination.\n");
	return;
  }

  /* BEHAVIOR ACCORDING DRAFT */

/* sending I1 message */
  if (is_local_addr(ipv6hdr, 8) && ipv6hdr[start+2] % ~(SHIM6_BITMASK_P) == SHIM6_TYPE_I1) {
	i1sent++; 

	/* store context information */
	if (i1sent <= 4) {
	  if (ctx == NULL) {
	    ctx = (struct shim_ctx *) malloc(sizeof(struct shim_ctx)); 
	    ctx->cstate = SHIM6_STATE_IDLE;
	  }
	  store_ctx_state(SHIM6_STATE_I1SENT);
	  store_ctx_ulidpair(ipv6hdr, start, 16);
	  store_ctx_fii(ipv6hdr, start, 16);
	  store_ctx_lp(ipv6hdr, 8, 24);
	  store_ctx_lslocal(stack);
	  store_ctx_taglocal(ipv6hdr, start+6);
	  store_ctx_nonceinit(ipv6hdr, start+12); 
	  store_reap_state(REAP_STATE_OPERATIONAL);
	  /* store struct shim_probe_list *rsent */
	  /* store struct shim_probe_list *rrecvd */
	}
	else {
	  i1sent = 0; 
	  /* failure */
	  store_ctx_state(SHIM6_STATE_EFAILED);
	  store_ctx_ulidpair(ipv6hdr, start, 16);
	  ctx->clocallp = NULL;  
	  ctx->cpeerlp = NULL;  
	  ctx->clocalls = NULL;
	  ctx->cpeerls = NULL;
	  ctx->cpeertag = NULL;
	  ctx->cnonceinit = NULL;
	  ctx->cnonceresp = NULL;
	  ctx->crespval = NULL;
	  ctx->crespvallen = 0;
	  store_reap_state(REAP_STATE_OPERATIONAL);
	}

        print_ctx(); 
	return;
  }
/* receiving I1 message */
  if (is_local_addr(ipv6hdr, 24) && ipv6hdr[start+2] % ~(SHIM6_BITMASK_P) == SHIM6_TYPE_I1) {

	for (j=0; j<4; j++) 
		nonce2[j] = rand();
	for (j=0; j<10; j++) /* a random responder validator, not a hash function */
		respval[j] = rand();
	memcpy(nonce1, &ipv6hdr[52], 4);

/*
	if (r1_received != 0) {
		sim_add_control_i2(pkt, &pkt_len, ctx->clocaltag, nonce2, ctx->cnonceresp, 18);

	        printf("Add ULID pair...\n\n"); 
		if (!lookup_ctx_lp(ctx->cpeerulid, ctx->clocalulid))
			sim_add_opts_ulidpair(pkt, &pkt_len, opt_critical, ctx->clocalulid, ctx->cpeerulid);

		printf("Add FII...\n\n"); 
		if (!lookup_ctx_fii(NULL)) {
			sim_add_opts_fii(pkt, &pkt_len, opt_critical, ctx->cfii);
		}

		printf("Store new info...\n\n"); 
		store_ctx_state(SHIM6_STATE_I2SENT);
		store_ctx_nonceinit(nonce2, 0); 
	
		printf("Add options...\n\n"); 
		sim_add_opts_loclist(pkt, &pkt_len, opt_critical, ctx->clocallistgen, ctx->clocalls);
		sim_add_opts_respval(pkt, &pkt_len, opt_critical, ctx->crespval, ctx->crespvallen);
	
		r1_received = 0;

		if (thc_generate_and_send_pkt(interface, NULL, NULL, pkt, &pkt_len) < 0) {
			fprintf(stdout, "Generating and sending packet failed\n");
		}
		return;
	}
*/
	/* match ctx ? */
	if (ctx == NULL) {
		printf("No context found...\n\n"); 

		/* send r1 */
		sim_add_control_r1(pkt, &pkt_len, nonce1, nonce2, 10);
		sim_add_opts_respval(pkt, &pkt_len, opt_critical, respval, 10); 
	}
	else {
		memcpy(ulidl, ctx->clocalulid, 16);
		memcpy(ulidp, ctx->cpeerulid, 16);

		if (!lookup_ctx_ulidpair(ipv6hdr, start, 16, ulidp, ulidl)) {
			/* send r1 */
			sim_add_control_r1(pkt, &pkt_len, nonce1, nonce2, 10);
			sim_add_opts_respval(pkt, &pkt_len, opt_critical, respval, 10); 
		}
		else if (ctx->cstate != SHIM6_STATE_EFAILED ||
			 ctx->cstate != SHIM6_STATE_NOSUPPORT) {
			/* send r1 */
			sim_add_control_r1(pkt, &pkt_len, nonce1, nonce2, 10);
			sim_add_opts_respval(pkt, &pkt_len, opt_critical, respval, 10); 
		}
		else if (ctx->cstate != SHIM6_STATE_ESTABLISHED) {
			printf("Non-established context found...\n\n"); 

			/* send r2 */
			sim_add_control_r2(pkt, &pkt_len, ctx->clocaltag, nonce1, 10);

			/* add locator list */
			sim_add_opts_loclist(pkt, &pkt_len, opt_critical, ctx->clocallistgen, ctx->clocalls);
		}
		else if (!is_peer_addr(ipv6hdr, 8)) {
			printf("Established context found, locator not in set...\n\n"); 

			/* send r1 */
			sim_add_control_r1(pkt, &pkt_len, nonce1, nonce2, 10);
			sim_add_opts_respval(pkt, &pkt_len, opt_critical, respval, 10); 
		}
		else if (!lookup_ctx_peertag(ipv6hdr, start+6)) {
			printf("Established context found, peer context tag does not match...\n\n"); 

			/* send r1 */
			sim_add_control_r1(pkt, &pkt_len, nonce1, nonce2, 10);
			sim_add_opts_respval(pkt, &pkt_len, opt_critical, respval, 10);
		}
		else {
			printf("Valid established context found...\n\n"); 

			/* send r2 */
			sim_add_control_r2(pkt, &pkt_len, ctx->clocaltag, nonce1, 10);
			/* add locator list */
			sim_add_opts_loclist(pkt, &pkt_len, opt_critical, ctx->clocallistgen, ctx->clocalls);
			memcpy(ctx->clocallistgen, listgen, 4); 
		}
	}

	/* send packet */
	if (thc_generate_and_send_pkt(interface, NULL, NULL, pkt, &pkt_len) < 0) {
		fprintf(stdout, "Generating and sending packet failed\n");
	}
	return;
  }
/* sending r1 messages */
  if (is_local_addr(ipv6hdr, 8) && ipv6hdr[start+2] % ~(SHIM6_BITMASK_P) == SHIM6_TYPE_R1) {
	if (ctx != NULL)
 	  print_ctx(); 
	return;	
  }
/* receiving r1 messages */
  if (is_local_addr(ipv6hdr, 24) && ipv6hdr[start+2] % ~(SHIM6_BITMASK_P) == SHIM6_TYPE_R1) {
	if (ctx == NULL) {
		printf("No context found...\n\n"); 
		return;	
	}
        
        printf("Context found...\n\n"); 

	memcpy(ulidl, &ipv6hdr[24], 16);
	memcpy(ulidp, &ipv6hdr[8], 16);

        printf("Check IPv6 addresses...\n\n"); 

	if (lp_in_ls(ulidp, ulidl) && lookup_ctx_nonceinit(ipv6hdr, start+8)) {
        	printf("Valid context found...\n\n"); 

		if (ctx->cstate == SHIM6_STATE_I1SENT) {
			printf("Context found and is in I1-Sent...\n\n"); 

			/* send i2 */
			
			/* include ulid pair, already done when creating ipv6 header */
			
			/* initiator nonce */
			for (j=0; j<4; j++) 
				nonce1[j] = rand();

		        printf("Add Nonce...\n\n"); 

			/* include responder nonce */
			memcpy(nonce2, &ipv6hdr[start+12], 4); 
			sim_add_control_i2(pkt, &pkt_len, ctx->clocaltag, nonce1, nonce2, 18);

		        printf("Add ULID pair...\n\n"); 

			if (!lookup_ctx_lp(ctx->cpeerulid, ctx->clocalulid))
				sim_add_opts_ulidpair(pkt, &pkt_len, opt_critical, ctx->clocalulid, ctx->cpeerulid);

		        printf("Add FII...\n\n"); 

			if (!lookup_ctx_fii(NULL)) {
				sim_add_opts_fii(pkt, &pkt_len, opt_critical, ctx->cfii);
			}

		        printf("Store new info...\n\n"); 

			/* store new information */
			store_ctx_state(SHIM6_STATE_I2SENT);
			store_ctx_nonceinit(nonce1, 0); 
			store_ctx_nonceresp(nonce2, 0); 
			store_ctx_respval(ipv6hdr, SHIM6_TYPE_R1, 40, 16); 
	
		        printf("Add options...\n\n"); 

			/* add locator list and respval */
			sim_add_opts_loclist(pkt, &pkt_len, opt_critical, ctx->clocallistgen, ctx->clocalls);
			sim_add_opts_respval(pkt, &pkt_len, opt_critical, ctx->crespval, ctx->crespvallen);
			
			/* 
			sim_add_opts_locpref(pkt, &pkt_len, opt_critical, ctx->clocallistgen, 3, ctx->clocalls); 
			sim_add_opts_cgapdm(pkt, &pkt_len, opt_critical, nonce1, 4);
			sim_add_opts_cgasig(pkt, &pkt_len, opt_critical, nonce2, 4);
			sim_add_opts_illegal(pkt, &pkt_len, opt_critical, nonce1, 4);
			*/
			
			/* send packet */
		
			/* check validator_min_lifetime, wait long time before reply i2 */
			/*
			sleep(VALIDATOR_MIN_LIFETIME);
			*/

			/* not sending to test receive i2 messages in i1-sent */
			if (thc_generate_and_send_pkt(interface, NULL, NULL, pkt, &pkt_len) < 0) {
				fprintf(stdout, "Generating and sending packet failed\n");
			}

			/* instead, set bool to trigger simulation trace */
			/* r1_received = 1; */

			return;
		}
		else {
			printf("Context found, but not in I1-Sent...\n\n"); 
			return;
		}
	}
 
	printf("No valid context found...\n\n"); 
	return;
  }
/* sending i2 messages */
  if (is_local_addr(ipv6hdr, 8) && ipv6hdr[start+2] % ~(SHIM6_BITMASK_P) == SHIM6_TYPE_I2) {
	print_ctx(); 
	return;		
  }
/* receiving i2 messages */
  if (is_local_addr(ipv6hdr, 24) && ipv6hdr[start+2] % ~(SHIM6_BITMASK_P) == SHIM6_TYPE_I2) {
	/* responder nonce and responder validator are not verified in this version */	

	if (ctx == NULL) {
    		ctx = (struct shim_ctx *) malloc(sizeof(struct shim_ctx)); 
		ctx->cstate = SHIM6_STATE_IDLE;
 	}

	/* to test the behaviour of receiving payload in i2-sent */

	store_ctx_tagpeer(ipv6hdr, start+6);
	sim_add_payload(pkt, &pkt_len, ctx->cpeertag); 


	/* send packet */
	if (thc_generate_and_send_pkt(interface, NULL, NULL, pkt, &pkt_len) < 0) {
		fprintf(stdout, "Generating and sending packet failed\n");
	}
 	return;

	/* allocate context */
	if (ctx->cstate == SHIM6_STATE_IDLE || 
	    ctx->cstate != SHIM6_STATE_EFAILED ||
	    ctx->cstate != SHIM6_STATE_NOSUPPORT) {

	  for (j=0; j<4; j++) 
		ctag[j] = rand();
	  memcpy(nonce1, &ipv6hdr[start+12], 4);	

	  store_ctx_state(SHIM6_STATE_ESTABLISHED);
	  store_ctx_ulidpair(ipv6hdr, start, 24);
	  store_ctx_lp(ipv6hdr, 8, 24);
	  store_ctx_lslocal(stack);
	  store_ctx_lspeer(ipv6hdr, SHIM6_TYPE_I2, start, 24);
	  store_ctx_taglocal(ctag, 0);
	  store_ctx_tagpeer(ipv6hdr, start+6);
	  store_reap_state(REAP_STATE_OPERATIONAL);
	  /* store struct shim_probe_list *rsent */
	  /* store struct shim_probe_list *rrecvd */

	  /* send r2 */
	  sim_add_control_r2(pkt, &pkt_len, ctx->cpeertag, nonce1, 10);
	  /* add locator list */	  
	  sim_add_opts_loclist(pkt, &pkt_len, opt_critical, ctx->clocallistgen, ctx->clocalls);

	  /* send packet */
	  if (thc_generate_and_send_pkt(interface, NULL, NULL, pkt, &pkt_len) < 0) {
		fprintf(stdout, "Generating and sending packet failed\n");
	  }
 	  return;
	}
	else if (ctx->cstate == SHIM6_STATE_I1SENT) {
		store_ctx_lspeer(ipv6hdr, SHIM6_TYPE_I2, 40, 24); 
		
		if (is_peer_addr(ipv6hdr, 8)) {
		  store_ctx_state(SHIM6_STATE_ESTABLISHED);
		  store_ctx_tagpeer(ipv6hdr, start+6);

		  /* send r2 */
		  sim_add_control_r2(pkt, &pkt_len, ctx->cpeertag, nonce1, 10);
		  /* add locator list */	  
		  sim_add_opts_loclist(pkt, &pkt_len, opt_critical, ctx->clocallistgen, ctx->clocalls);
		}
	}
	else if (ctx->cstate == SHIM6_STATE_I2SENT ||
		 ctx->cstate != SHIM6_STATE_I2BISSENT ||
		 ctx->cstate != SHIM6_STATE_ESTABLISHED) {

		store_ctx_lspeer(ipv6hdr, SHIM6_TYPE_I2, start, 24);

		if (is_peer_addr(ipv6hdr, 8)) {
		  store_ctx_tagpeer(ipv6hdr, start+6);

		  /* send r2 */
		  sim_add_control_r2(pkt, &pkt_len, ctx->cpeertag, nonce1, 10);
		  /* add locator list */	  
		  sim_add_opts_loclist(pkt, &pkt_len, opt_critical, ctx->clocallistgen, ctx->clocalls);
		}	
	}

	return;
  }
/* sending r2 messages */
  if (is_local_addr(ipv6hdr, 8) && ipv6hdr[start+2] % ~(SHIM6_BITMASK_P) == SHIM6_TYPE_R2) {
	if (ctx != NULL)
 	  print_ctx(); 

	/* check for context confusion */

	return;	
  }
/* receiving r2 messages */
  if (is_local_addr(ipv6hdr, 24) && ipv6hdr[start+2] % ~(SHIM6_BITMASK_P) == SHIM6_TYPE_R2) {
	if (ctx == NULL) {
		printf("No context found...\n\n");
		return;	
	}

	if (ctx->cstate == SHIM6_STATE_I1SENT ||	
	    ctx->cstate == SHIM6_STATE_I2SENT ||	
	    ctx->cstate == SHIM6_STATE_I2BISSENT) {

		memcpy(ulidl, &ipv6hdr[24], 16);
		memcpy(ulidp, &ipv6hdr[8], 16);

		if (lookup_ctx_lp(ulidp, ulidl) && lookup_ctx_nonceinit(ipv6hdr, start+12)) {
		  store_ctx_state(SHIM6_STATE_ESTABLISHED);
		  store_ctx_lspeer(ipv6hdr, SHIM6_TYPE_R2, start, 16);
		  store_ctx_tagpeer(ipv6hdr, start+6);

	 	  print_ctx(); 

		  /* to test shim6 payload */
		  /*
		  sim_add_payload(pkt, &pkt_len, ctx->clocaltag); 
		  */

		  /* to test locator preferences: */
		  /*
		  for (j=0; j<4; j++) 
			nonce1[j] = rand();
		  sim_add_control_i2(pkt, &pkt_len, ctx->clocaltag, ctx->cnonceinit, ctx->cnonceresp, 18);
		  sim_add_opts_loclist(pkt, &pkt_len, opt_critical, ctx->clocallistgen, ctx->clocalls);
		  sim_add_opts_locpref(pkt, &pkt_len, opt_critical, ctx->clocallistgen, 3, ctx->clocalls); 
		  */
		
		  /* send packet */
		  
		  if (thc_generate_and_send_pkt(interface, NULL, NULL, pkt, &pkt_len) < 0) {
		  	fprintf(stdout, "Generating and sending packet failed\n");
		  }
		  
		  return;
		}
	}

	return;		
  }

/* sending r1bis messages */
/* receiving r1bis messages */
/* sending i2bis messages */
/* receiving i2bis messages */
/* sending update request messages */
  if (is_local_addr(ipv6hdr, 8) && ipv6hdr[start+2] % ~(SHIM6_BITMASK_P) == SHIM6_TYPE_UPDREQ) {
	return;	
  }
/* receiving update request messages */
  if (is_local_addr(ipv6hdr, 24) && ipv6hdr[start+2] % ~(SHIM6_BITMASK_P) == SHIM6_TYPE_UPDREQ) {
	return;	
  }
/* sending update acknowledgment messages */
  if (is_local_addr(ipv6hdr, 8) && ipv6hdr[start+2] % ~(SHIM6_BITMASK_P) == SHIM6_TYPE_UPDACK) {
	return;	
  }
/* receiving update acknowledgment messages */
  if (is_local_addr(ipv6hdr, 24) && ipv6hdr[start+2] % ~(SHIM6_BITMASK_P) == SHIM6_TYPE_UPDACK) {
	return;	
  }

  return;
}

/* MAIN PROGRAM */

int main(int argc, char **argv) {
    int i; 
    char *ulid_local;

    /* print tool information */
    fprintf(stdout, "Shimterceptor: A User Level Shim6 Interceptor (c) 2007\n\n");

    if (argc < 2) {
        return help(argv[0]);
    }
    else if (strncmp(argv[1], "-help", 5) == 0) {
        return help(argv[0]);
    }

    /* locators */
    i = 1;
    while (i<argc) {
	/* add locator */
	ulid_local = malloc(16);
	ulid_local = thc_get_own_ipv6(argv[i], NULL, PREFER_GLOBAL);
	add_locator((unsigned char *) ulid_local);
	i++;
    }

    /* spoofed locators */
    /*
    add_locator((unsigned char *) thc_string2ipv6((unsigned char *) multicast));
    add_locator((unsigned char *) thc_string2ipv6((unsigned char *) spoof));
    */

    interface = argv[1]; 
    my_ipv6 = thc_get_own_ipv6(interface, NULL, PREFER_GLOBAL);

    /* capture received packets */
    fprintf(stdout, "Start intercepting Shim6 messages. Press Control-C to end...\n"); 
    return thc_pcap_function(interface, "ip6", (char *) intercept);
}
